package memory

import (
	"sync"

	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/golibs/uuid"
)

var _ model.Store = &store{}

type store struct {
	uuid                   uuid.Source
	categories             map[string]*model.Category
	categoriesBySlug       map[string]*model.Category
	languageCodes          map[string]*model.CategoryLanguageCode
	translations           map[string]map[string]*model.CategoryTranslation
	categoriesToExtensions map[string][]*model.CuratedCategoryExtension
	extensionsToCategories map[string][]*model.CuratedCategoryExtension

	featuredCarouselSchedules       map[string]*model.FeaturedCarouselSchedule
	featuredCarouselSchedulesBySlug map[string]*model.FeaturedCarouselSchedule
	featuredCarousels               map[string]*model.FeaturedCarousel
	scheduleIDsToCarousels          map[string]*model.FeaturedCarousel
	featuredCarouselEntries         map[string]*model.FeaturedCarouselEntry
	carouselsToEntries              map[string][]*model.FeaturedCarouselEntry

	// Maps extension ID to map of version IDs to extension discovery data for that extension/version
	extensionsDiscoveryData map[string]map[string]*model.ExtensionDiscoveryData

	resetEnabled bool
	mutex        sync.Mutex
}

func New(uuid uuid.Source) model.Store {
	return &store{
		uuid:                            uuid,
		categories:                      make(map[string]*model.Category),
		categoriesBySlug:                make(map[string]*model.Category),
		languageCodes:                   make(map[string]*model.CategoryLanguageCode),
		translations:                    make(map[string]map[string]*model.CategoryTranslation),
		categoriesToExtensions:          make(map[string][]*model.CuratedCategoryExtension),
		extensionsToCategories:          make(map[string][]*model.CuratedCategoryExtension),
		featuredCarouselSchedules:       make(map[string]*model.FeaturedCarouselSchedule),
		featuredCarouselSchedulesBySlug: make(map[string]*model.FeaturedCarouselSchedule),
		featuredCarousels:               make(map[string]*model.FeaturedCarousel),
		scheduleIDsToCarousels:          make(map[string]*model.FeaturedCarousel),
		featuredCarouselEntries:         make(map[string]*model.FeaturedCarouselEntry),
		carouselsToEntries:              make(map[string][]*model.FeaturedCarouselEntry),
		extensionsDiscoveryData:         make(map[string]map[string]*model.ExtensionDiscoveryData),
	}
}

func (s *store) IsResetEnabled() bool {
	defer s.mutex.Unlock()
	s.mutex.Lock()
	return s.resetEnabled
}

func (s *store) EnableDataReset() {
	defer s.mutex.Unlock()
	s.mutex.Lock()
	s.resetEnabled = true
}

func (s *store) ResetAllData() error {
	defer s.mutex.Unlock()
	s.mutex.Lock()
	if !s.resetEnabled {
		return data.ErrUnavailable
	}
	s.categories = make(map[string]*model.Category)
	s.categoriesBySlug = make(map[string]*model.Category)
	s.languageCodes = make(map[string]*model.CategoryLanguageCode)
	s.translations = make(map[string]map[string]*model.CategoryTranslation)
	s.categoriesToExtensions = make(map[string][]*model.CuratedCategoryExtension)
	s.extensionsToCategories = make(map[string][]*model.CuratedCategoryExtension)
	s.featuredCarouselSchedules = make(map[string]*model.FeaturedCarouselSchedule)
	s.featuredCarouselSchedulesBySlug = make(map[string]*model.FeaturedCarouselSchedule)
	s.featuredCarousels = make(map[string]*model.FeaturedCarousel)
	s.featuredCarouselEntries = make(map[string]*model.FeaturedCarouselEntry)
	s.carouselsToEntries = make(map[string][]*model.FeaturedCarouselEntry)
	s.extensionsDiscoveryData = make(map[string]map[string]*model.ExtensionDiscoveryData)

	return nil
}

func (s *store) getOrCreateLanguageCode(language string) (*model.CategoryLanguageCode, error) {
	var languageCode *model.CategoryLanguageCode
	for _, val := range s.languageCodes {
		if val.Code == language {
			languageCode = val
			break
		}
	}
	if languageCode == nil {
		id, err := s.uuid.Next()
		if err != nil {
			return nil, err
		}
		languageCode = &model.CategoryLanguageCode{
			ID:   id,
			Code: language,
		}
		s.languageCodes[id] = languageCode
	}
	return languageCode, nil
}
