package postgres

import (
	"fmt"
	"reflect"
	"sync"
	"sync/atomic"

	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/migrations"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/golibs/logger"

	"github.com/jinzhu/gorm"
	_ "github.com/jinzhu/gorm/dialects/postgres"
	"github.com/pkg/errors"

	"gopkg.in/gormigrate.v1"
)

var (
	_ model.Store = &store{}
)

type store struct {
	db              *gorm.DB
	logger          logger.Logger
	languageCodeIDs map[string]string

	resetEnabled int32
}

var allModels = []interface{}{
	&model.Category{},
	&model.CategoryLanguageCode{},
	&model.CategoryTranslation{},
	&model.CuratedCategoryExtension{},
	&model.FeaturedCarouselEntry{},
	&model.FeaturedCarousel{},
	&model.FeaturedCarouselSchedule{},
	&model.FeaturedCarouselDefault{},
	&model.ExtensionDiscoveryData{},
}

func New(host string, port int, username, password, dbname, prefix string, logger logger.Logger) (model.Store, error) {
	if logger == nil {
		return nil, errors.New("logger cannot be nil")
	}
	gorm.DefaultTableNameHandler = func(db *gorm.DB, defaultTableName string) string {
		return prefix + "_" + defaultTableName
	}

	configStr := fmt.Sprintf("host=%s port=%d user=%s password=%s dbname=%s sslmode=disable", host, port, username, password, dbname)
	db, err := gorm.Open("postgres", configStr)
	if err != nil {
		return nil, errors.Wrap(err, "Could not open gorm")
	}

	db = db.Set("table_prefix", prefix)

	discoDB := store{
		db:              db,
		logger:          logger,
		languageCodeIDs: map[string]string{},
	}

	m := gormigrate.New(db, &gormigrate.Options{
		TableName:      prefix + "_" + "migrations",
		IDColumnName:   "id",
		IDColumnSize:   255,
		UseTransaction: true,
	}, []*gormigrate.Migration{
		migrations.CategoriesSetupTables,
		migrations.CategoriesAddForeignKeys,
		migrations.FeaturedCarouselsRefactor,
	})
	err = m.Migrate()
	if err != nil {
		return nil, errors.Wrap(err, "could not migrate")
	}

	m = gormigrate.New(db, &gormigrate.Options{
		TableName:      prefix + "_" + "migrations",
		IDColumnName:   "id",
		IDColumnSize:   255,
		UseTransaction: true,
	}, []*gormigrate.Migration{
		migrations.CategoriesAddSlugs,
		migrations.CarouselEntriesRemoveForeignKey,
		migrations.SchedulesAddSlugs,
	})
	err = m.Migrate()
	if err != nil {
		return nil, errors.Wrap(err, "could not migrate")
	}

	m = gormigrate.New(db, &gormigrate.Options{
		TableName:      prefix + "_" + "migrations",
		IDColumnName:   "id",
		IDColumnSize:   255,
		UseTransaction: true,
	}, []*gormigrate.Migration{
		migrations.CarouselEntriesAddForeignKey,
		migrations.FeaturedCarouselsEntryContent,
	})
	err = m.Migrate()
	if err != nil {
		return nil, errors.Wrap(err, "could not migrate")
	}

	m = gormigrate.New(db, &gormigrate.Options{
		TableName:      prefix + "_" + "migrations",
		IDColumnName:   "id",
		IDColumnSize:   255,
		UseTransaction: true,
	}, []*gormigrate.Migration{
		migrations.DiscoveryAddHstore,
		migrations.DiscoverySetupTables,
	})
	err = m.Migrate()
	if err != nil {
		return nil, errors.Wrap(err, "could not migrate")
	}

	for _, code := range model.SupportedLanguageCodes {
		var clc model.CategoryLanguageCode
		err = db.FirstOrCreate(&clc, model.CategoryLanguageCode{Code: code}).Error
		if err != nil {
			return nil, errors.Wrap(err, "Error in FirstOrCreate")
		}
		discoDB.languageCodeIDs[clc.Code] = clc.ID
	}

	return &discoDB, nil
}

func (s *store) IsResetEnabled() bool { return atomic.LoadInt32(&s.resetEnabled) != 0 }
func (s *store) EnableDataReset()     { atomic.StoreInt32(&s.resetEnabled, 1) }

func (s *store) ResetAllData() error {
	if !s.IsResetEnabled() {
		return errors.Wrap(data.ErrUnavailable, "Reset is not enabled")
	}

	var active sync.WaitGroup
	// len(allModels) - 1 because we skip over model.CategoryLanguageCode
	active.Add(len(allModels) - 1)

	concurrentErr := struct {
		err error
		sync.Mutex
	}{}

	for _, mod := range allModels {
		r := reflect.ValueOf(mod)
		if r.Type() == reflect.TypeOf(&model.CategoryLanguageCode{}) {
			// don't blow away language codes, we just created them and they're important.
			continue
		}
		go func(m interface{}) {
			defer active.Done()
			if err := s.db.Delete(m).Error; err != nil {
				concurrentErr.Lock()
				concurrentErr.err = errors.Wrapf(err, "Could not delete model %v", reflect.TypeOf(m).Name())
				concurrentErr.Unlock()
			}
		}(mod)
	}
	active.Wait()
	return concurrentErr.err
}

func (s *store) getLanguageCodeID(language string) string {
	if id, ok := s.languageCodeIDs[language]; ok {
		return id
	}

	return s.languageCodeIDs[model.DefaultCategoryLanguageCode]
}

func (s *store) getTranslation(language, cid string) (*model.CategoryTranslation, error) {
	targetID := s.getLanguageCodeID(language)

	var translations []*model.CategoryTranslation
	err := s.db.
		Where(&model.CategoryTranslation{
			CategoryID:     cid,
			LanguageCodeID: targetID,
		}).
		Or(&model.CategoryTranslation{
			CategoryID:     cid,
			LanguageCodeID: s.languageCodeIDs[model.DefaultCategoryLanguageCode],
		}).
		Find(&translations).Error

	if err != nil {
		if gorm.IsRecordNotFoundError(err) {
			return nil, nil
		}

		return nil, errors.Wrap(err, "could not get category translation")
	}

	if len(translations) == 1 {
		return translations[0], nil
	}

	for _, t := range translations {
		if t.LanguageCodeID == targetID {
			return t, nil
		}
	}

	return nil, nil
}
