package model

import (
	discovery "code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/twirputils"
	"github.com/pkg/errors"
)

// BuildCategoryDocument creates a twirp category document from a model.Category and its translations
func BuildCategoryDocument(category *Category, translation *CategoryTranslation) *discovery.CategoryDocument {
	if category == nil {
		return &discovery.CategoryDocument{}
	}
	if translation == nil {
		// this is not an error; this can happen, for example, on category creation.
		// This way the caller doesn't have to make the fake translation for us,
		// we can just do it here.
		translation = MakeMissingCategoryTranslation()
	}
	return &discovery.CategoryDocument{
		ID:          category.ID,
		Type:        category.Type,
		Order:       category.Order,
		Slug:        twirputils.WrapStringPointer(category.Slug),
		SortKey:     category.SortKey,
		Readonly:    category.Readonly,
		Visible:     category.Visible,
		Deleted:     category.Deleted,
		Name:        translation.Name,
		Description: translation.Description,
	}
}

// BuildCategoriesDocument creates a twirp categories document from provided lists of model.Category and associated translations
func BuildCategoriesDocument(categories []*Category, translations []*CategoryTranslation, totalCategories int32) (*discovery.CategoriesDocument, error) {
	if len(categories) != len(translations) {
		return nil, ErrTranslationCountMismatch
	}
	resp := discovery.CategoriesDocument{
		Count:      totalCategories,
		Categories: make([]*discovery.CategoryDocument, 0, len(categories)),
	}

	for i, cat := range categories {
		translation := translations[i]
		twirpCat := BuildCategoryDocument(cat, translation)
		resp.Categories = append(resp.Categories, twirpCat)
	}

	return &resp, nil
}

// BuildCategoryOrderDocument creates a twirp category order document from a provided list of model.Category
func BuildCategoryOrderDocument(categories []*Category) (*discovery.CategoryOrderDocument, error) {
	resp := discovery.CategoryOrderDocument{
		CategoryOrder: make([]*discovery.CategoryDocument, 0, len(categories)),
	}
	for _, cat := range categories {
		entry := discovery.CategoryDocument{
			ID:    cat.ID,
			Order: cat.Order,
		}
		resp.CategoryOrder = append(resp.CategoryOrder, &entry)
	}
	return &resp, nil
}

// BuildFeaturedCarouselEntryDocument creates a twirp document from a model carousel entry
func BuildFeaturedCarouselEntryDocument(entry *FeaturedCarouselEntry) (*discovery.FeaturedCarouselEntryDocument, error) {
	return &discovery.FeaturedCarouselEntryDocument{
		ID:         entry.ID,
		CarouselID: entry.CarouselID,
		Content: &discovery.FeaturedCarouselEntryContent{
			ClickThrough: entry.ClickThrough,
			ImageURL:     entry.ImageURL,
			Title:        entry.Title,
		},
		Order: entry.Order,
	}, nil
}

// BuildFeaturedCarouselEntriesDocument creates a twirp document from a list of model carousel entries
func BuildFeaturedCarouselEntriesDocument(entries []*FeaturedCarouselEntry, totalEntries int32) (*discovery.FeaturedCarouselEntriesDocument, error) {
	doc := discovery.FeaturedCarouselEntriesDocument{
		Count:           totalEntries,
		CarouselEntries: make([]*discovery.FeaturedCarouselEntryDocument, len(entries)),
	}

	for i, entry := range entries {
		entryDoc, err := BuildFeaturedCarouselEntryDocument(entry)
		if err != nil {
			return nil, err
		}
		doc.CarouselEntries[i] = entryDoc
	}

	return &doc, nil
}

// BuildFeaturedCarouselDocument creates a twirp document from a model featured carousel
func BuildFeaturedCarouselDocument(carousel *FeaturedCarousel, entries []*FeaturedCarouselEntry, totalEntries int32) (*discovery.FeaturedCarouselDocument, error) {
	entriesDoc, err := BuildFeaturedCarouselEntriesDocument(entries, totalEntries)
	if err != nil {
		return nil, err
	}

	startTime, err := twirputils.WrapTimePointer(&carousel.StartTime)
	if err != nil {
		return nil, errors.Wrap(err, "")
	}

	return &discovery.FeaturedCarouselDocument{
		ID:         carousel.ID,
		ScheduleID: carousel.ScheduleID,
		StartTime:  startTime,
		Entries:    entriesDoc,
	}, nil
}

// BuildFeaturedCarouselsDocument creates a twirp document given a list of model carousels and their associated model entries
func BuildFeaturedCarouselsDocument(carousels []*FeaturedCarousel, entriesList [][]*FeaturedCarouselEntry, totalCarousels int32) (*discovery.FeaturedCarouselsDocument, error) {
	count := len(carousels)
	carouselDocs := make([]*discovery.FeaturedCarouselDocument, count)

	for i := 0; i < count; i++ {
		carousel := carousels[i]
		entries := entriesList[i]
		carouselDoc, err := BuildFeaturedCarouselDocument(carousel, entries, int32(len(entries)))
		if err != nil {
			return nil, err
		}
		carouselDocs[i] = carouselDoc
	}

	return &discovery.FeaturedCarouselsDocument{
		Count:     totalCarousels,
		Carousels: carouselDocs,
	}, nil
}

// BuildFeaturedScheduleDocument creates a twirp document given a model featured carousel schedule and its associated current carousel and entries
func BuildFeaturedScheduleDocument(schedule *FeaturedCarouselSchedule, carousel *FeaturedCarousel, entries []*FeaturedCarouselEntry) (*discovery.FeaturedScheduleDocument, error) {
	var carouselDoc *discovery.FeaturedCarouselDocument
	var err error
	if carousel != nil {
		carouselDoc, err = BuildFeaturedCarouselDocument(carousel, entries, int32(len(entries)))
		if err != nil {
			return nil, err
		}
	}

	return &discovery.FeaturedScheduleDocument{
		ID:              schedule.ID,
		Description:     schedule.Description,
		Slug:            twirputils.WrapStringPointer(schedule.Slug),
		CurrentCarousel: carouselDoc,
	}, nil
}

// BuildFeaturedSchedulesDocument creates a twirp document given a list of model schedules and associated carousels and carousel entries
func BuildFeaturedSchedulesDocument(schedules []*FeaturedCarouselSchedule, totalSchedules int32, carousels []*FeaturedCarousel, entries [][]*FeaturedCarouselEntry) (*discovery.FeaturedSchedulesDocument, error) {
	count := len(schedules)
	scheduleDocs := make([]*discovery.FeaturedScheduleDocument, count)

	for i := 0; i < count; i++ {
		scheduleDoc, err := BuildFeaturedScheduleDocument(schedules[i], carousels[i], entries[i])
		if err != nil {
			return nil, err
		}
		scheduleDocs[i] = scheduleDoc
	}

	return &discovery.FeaturedSchedulesDocument{
		Count:     totalSchedules,
		Schedules: scheduleDocs,
	}, nil
}

// BuildExtensionVersionDiscoveryDocument creates a twirp document given a discovery data object
func BuildExtensionVersionDiscoveryDocument(discoData *ExtensionDiscoveryData) (*discovery.ExtensionVersionDiscoveryDocument, error) {
	return &discovery.ExtensionVersionDiscoveryDocument{
		ExtensionID: discoData.ExtensionID,
		Version:     discoData.Version,
		DiscoveryData: &discovery.DiscoveryDataDocument{
			AuthorName:       discoData.AuthorName,
			Categories:       discoData.Categories,
			Description:      discoData.Description,
			EULATOSURL:       discoData.EULATOSURL,
			Games:            discoData.Games,
			IconPaths:        HstoreToMap(discoData.IconPaths),
			IconURLs:         HstoreToMap(discoData.IconURLs),
			Name:             discoData.Name,
			PrivacyPolicyURL: discoData.PrivacyPolicyURL,
			ScreenshotPaths:  discoData.ScreenshotPaths,
			ScreenshotURLs:   discoData.ScreenshotURLs,
			Summary:          discoData.Summary,
			SupportEmail:     discoData.SupportEmail,
			ViewerSummary:    discoData.ViewerSummary,
			Deleted:          !discoData.DeletedAt.IsZero(),
		},
	}, nil
}
