package data

import (
	"net/mail"
	"net/url"

	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/sliceutils"
)

const (
	MaxEmailLength = 254
)

var (
	validCategoryTypes = []string{
		model.CategoryTypeCurated,
		model.CategoryTypeDeveloper,
		model.CategoryTypePseudo,
	}

	validSortKeys = []string{
		model.CategorySortkeyManual,
		model.CategorySortkeyPopularity,
		model.CategorySortkeyTime,
	}
)

func ValidateCategoryUpsert(cat *discovery.CategoryUpsert, allowEmpty bool) error {
	if (!allowEmpty && cat != nil && cat.GetType() == nil) ||
		(cat.GetType() != nil && !sliceutils.In(cat.GetType().GetValue(), validCategoryTypes)) {
		return ErrUnknownCategoryType
	}

	if (!allowEmpty && cat != nil && cat.GetSortKey() == nil) ||
		(cat.GetSortKey() != nil && !sliceutils.In(cat.GetSortKey().GetValue(), validSortKeys)) {
		return ErrUnknownCategorySortKey
	}

	if cat != nil && cat.GetType() != nil && cat.GetType().GetValue() == model.CategoryTypePseudo {
		return validatePseudoCategory(cat)
	}

	return nil
}

// ValidateCategoryUpdate ensures that the provided upsert can be applied to a given category
// It requires loading the model from the data store and therefore cannot be done purely on the request
func ValidateCategoryUpdate(cat *model.Category, upsert *discovery.CategoryUpsert) error {
	if cat == nil {
		return ErrNotFound
	}
	if cat.Deleted ||
		(upsert.GetType() != nil && cat.Type != upsert.GetType().GetValue()) {
		return ErrInvalidCategoryUpdate
	}
	return nil
}

func validatePseudoCategory(cat *discovery.CategoryUpsert) error {
	if cat.GetVisible() != nil && cat.GetVisible().GetValue() {
		return ErrInvalidCategoryUpdate
	}

	if cat.GetReadonly() != nil && !cat.GetReadonly().GetValue() {
		return ErrInvalidCategoryUpdate
	}

	if cat.GetOrder() != nil && cat.GetOrder().GetValue() != 0 {
		return ErrInvalidCategoryUpdate
	}

	if cat.GetSlug() == nil ||
		(cat.GetSlug() != nil && cat.GetSlug().GetValue() == "") {
		return ErrRequiresSlug
	}

	return nil
}

func ValidateAddCategory(req *discovery.AddCategoryRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetCategory() == nil {
		return ErrMissingParameter
	}

	return nil
}

func ValidateAddCategoryToExtension(req *discovery.AddCategoryToExtensionRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetCategoryID() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateAddExtensionToCategory(req *discovery.AddExtensionToCategoryRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetExtensionID() == "" {
		return ErrMissingParameter
	}
	if req.GetVersionID() == nil || req.GetVersionID().GetValue() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateAddFeaturedCarousel(req *discovery.AddFeaturedCarouselRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.FeaturedCarousel == nil || req.FeaturedCarousel.StartTime == nil {
		return ErrMissingParameter
	}

	return nil
}

func ValidateAddFeaturedCarouselEntry(req *discovery.AddFeaturedCarouselEntryRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetFeaturedCarouselEntry() == nil {
		return ErrMissingParameter
	}
	if req.GetFeaturedCarouselEntry().GetCarouselID() == nil {
		return ErrMissingEntryCarouselID
	}
	if req.GetFeaturedCarouselEntry().GetContent() == nil {
		return ErrMissingEntryContent
	}

	return nil
}

func ValidateAddFeaturedSchedule(req *discovery.AddFeaturedScheduleRequest) error {
	if req == nil {
		return ErrMissingParameter
	}

	return nil
}

func ValidateAddGameToExtension(req *discovery.AddGameToExtensionRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	// Game ID is not an optional parameter but if it's set to the default value
	// don't allow that (since no game has an ID of 0).  Don't want to change to a
	// pointer and then check for nil for EMS API compatibility.
	if req.GetGameID() == 0 {
		return ErrInvalidGameID
	}

	return nil
}

func ValidateDeleteCategory(req *discovery.DeleteCategoryRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetCategoryID() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateDeleteExtensionVersionDiscoveryData(req *discovery.DeleteExtensionVersionDiscoveryDataRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetExtensionID() == "" {
		return ErrMissingParameter
	}
	if req.GetVersion() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateHardDeleteExtensionDiscoveryData(req *discovery.HardDeleteExtensionDiscoveryDataRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetExtensionID() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateDeleteFeaturedCarousel(req *discovery.DeleteFeaturedCarouselRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetCarouselID() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateDeleteFeaturedCarouselEntry(req *discovery.DeleteFeaturedCarouselEntryRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetID() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateDeleteFeaturedSchedule(req *discovery.DeleteFeaturedScheduleRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetScheduleID() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateDeleteGameFromExtension(req *discovery.DeleteGameFromExtensionRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetGameID() <= 0 {
		return ErrInvalidGameID
	}

	return nil
}

func ValidateEditCategoryTranslation(req *discovery.EditCategoryTranslationRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetCategoryID() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateGetCategories(req *discovery.GetCategoriesRequest) error {
	if req == nil {
		return ErrMissingParameter
	}

	return nil
}

func ValidateGetCategory(req *discovery.GetCategoryRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetCategoryID() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateGetCategoryExtensions(req *discovery.GetCategoryExtensionsRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetCategoryID() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateGetExtensionVersionDiscoveryData(req *discovery.GetExtensionVersionDiscoveryDataRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetExtensionID() == "" {
		return ErrMissingParameter
	}
	if req.GetVersion() == "" {
		return ErrMissingParameter
	}

	return nil

}

func ValidateGetFeaturedCarousel(req *discovery.GetFeaturedCarouselRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetCarouselID() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateGetFeaturedCarouselEntries(req *discovery.GetFeaturedCarouselEntriesRequest) error {
	if req == nil {
		return ErrMissingParameter
	}

	return nil
}

func ValidateGetFeaturedCarouselEntry(req *discovery.GetFeaturedCarouselEntryRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetID() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateGetFeaturedCarousels(req *discovery.GetFeaturedCarouselsRequest) error {
	if req == nil {
		return ErrMissingParameter
	}

	return nil
}

func ValidateGetFeaturedSchedule(req *discovery.GetFeaturedScheduleRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetScheduleID() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateGetFeaturedSchedules(req *discovery.GetFeaturedSchedulesRequest) error {
	if req == nil {
		return ErrMissingParameter
	}

	return nil
}

func ValidateOrderCategories(req *discovery.OrderCategoriesRequest) error {
	if req == nil {
		return ErrMissingParameter
	}

	return nil
}

func ValidateOrderCategory(req *discovery.OrderCategoryRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetCategoryID() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidateOrderFeaturedCarousel(req *discovery.OrderFeaturedCarouselRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	if req.GetCarouselID() == "" {
		return ErrMissingParameter
	}

	return nil
}

func ValidatePutExtensionVersionDiscoveryData(req *discovery.PutExtensionVersionDiscoveryDataRequest) error {
	if req == nil {
		return ErrMissingParameter
	}
	reqData := req.GetUpdateData()
	if reqData == nil {
		return ErrMissingParameter
	}
	if reqData.GetExtensionID() == "" {
		return ErrMissingParameter
	}
	if reqData.GetVersion() == "" {
		return ErrMissingParameter
	}
	if reqData.GetName() != nil {
		name := reqData.GetName().GetValue()
		if len(name) > MaxNameLength {
			return ErrNameLengthLong
		}
		if name == "" {
			return ErrNameLengthShort
		}
	}
	if reqData.GetDescription() != nil && len(reqData.GetDescription().GetValue()) > MaxDescriptionLength {
		return ErrDescriptionLength
	}
	if reqData.GetSummary() != nil && len(reqData.GetSummary().GetValue()) > MaxSummaryLength {
		return ErrSummaryLength
	}
	if reqData.GetSupportEmail() != nil {
		email := reqData.GetSupportEmail().GetValue()
		if email == "" {
			return ErrInvalidSupportEmail
		}
		if !IsEmailValid(&email) {
			return ErrInvalidSupportEmail
		}
	}
	if reqData.GetEULATOSURL() != nil {
		tosURL := reqData.GetEULATOSURL().GetValue()
		if tosURL == "" {
			return ErrInvalidTermsURL
		}
		if !IsVAbsoluteURLValid(&tosURL) {
			return ErrInvalidTermsURL
		}
	}
	if reqData.GetPrivacyPolicyURL() != nil {
		privacyURL := reqData.GetPrivacyPolicyURL().GetValue()
		if privacyURL == "" {
			return ErrInvalidPrivacyURL
		}
		if !IsVAbsoluteURLValid(&privacyURL) {
			return ErrInvalidPrivacyURL
		}
	}
	if reqData.GetAuthorName() != nil && len(reqData.GetAuthorName().GetValue()) > MaxAuthorNameLength {
		return ErrAuthorNameLength
	}

	return nil
}

func ValidateRemoveExtensionFromCategory(req *discovery.RemoveExtensionFromCategoryRequest) error {
	if req == nil {
		return ErrMissingParameter
	}

	return nil
}

func ValidateUpdateCategory(req *discovery.UpdateCategoryRequest) error {
	if req == nil {
		return ErrMissingParameter
	}

	return nil
}

func ValidateUpdateFeaturedCarousel(req *discovery.UpdateFeaturedCarouselRequest) error {
	if req == nil {
		return ErrMissingParameter
	}

	return nil
}

func ValidateUpdateFeaturedCarouselEntry(req *discovery.UpdateFeaturedCarouselEntryRequest) error {
	if req == nil {
		return ErrMissingParameter
	}

	return nil
}

func ValidateUpdateFeaturedSchedule(req *discovery.UpdateFeaturedScheduleRequest) error {
	if req == nil {
		return ErrMissingParameter
	}

	return nil
}

func IsEmailValid(email *string) bool {
	if email == nil {
		return false
	}

	if len(*email) > MaxEmailLength {
		return false
	}

	address, err := mail.ParseAddress(*email)
	if err != nil || address == nil || address.Address != *email {
		return false
	}

	return true
}

// IsVAbsoluteURLValid indicates if the provided URL is an absolute URL and valid
// Might change depending on the outcome of https://jira.twitch.com/browse/DEVS-1162
func IsVAbsoluteURLValid(input *string) bool {
	if input == nil {
		return false
	}

	u, err := url.Parse(*input)
	if err != nil {
		return false
	}

	if !u.IsAbs() {
		return false
	}

	if u.Scheme != "http" && u.Scheme != "https" {
		return false
	}

	if u.User != nil {
		return false
	}

	return true
}
