package simplebootstrap

import (
	"errors"

	"code.justin.tv/amzn/C7-go/c7"
	"code.justin.tv/amzn/C7-go/resolvers"
	"code.justin.tv/amzn/TwitchLogging"
	"code.justin.tv/amzn/TwitchLoggingMiddleware"
	"code.justin.tv/amzn/TwitchProcessIdentifier"
	"code.justin.tv/amzn/TwitchTelemetry"
	metricsmiddleware "code.justin.tv/amzn/TwitchTelemetryMetricsMiddleware"
	"code.justin.tv/video/metricmiddleware-beta/operation"
)

// FultonBootstrap is the root configuration object which is handed to an application to startup
//   this struct provides core features such as logging, config, and processAddress
type FultonBootstrap struct {
	C7                *c7.C7
	Logger            logging.Logger
	OperationStarter  *operation.Starter
	ProcessIdentifier *identifier.ProcessIdentifier
	SampleReporter    telemetry.SampleReporter
}

// ConfigSetLoader a function that returns the config set for the bootstrapping
//   Generally you should just pass in DefaultConfigSetLoader
type ConfigSetLoader func() (*c7.C7Set, error)

// DefaultConfigSetLoader loads the configset that the fulton framework makes for you
// eg all.c7...
func DefaultConfigSetLoader() (*c7.C7Set, error) {
	return resolvers.ResolveFile("all.c7")
}

// FultonCustomMembers allows customers to pass in their preferred interface implementations for the bootstrap
type FultonCustomMembers struct {
	// Logger to use (recommended to be a structured logger, such as a logger that outputs JSON)
	Logger logging.Logger
	// SampleObserver to use, can be either buffered or unbuffered depending on implementations
	SampleObserver telemetry.SampleObserver
	// AutoFlushMetrics represents whether to flush metrics every request, recommended true for lambda and false for
	// anything else (false = batch metrics across calls, true = make metric network calls for every incoming request,
	// blocking on network calls)
	AutoFlushMetrics bool
}

// CommonSimpleBootstrap represents the bootstrap process that all Fulton services (Lambda, ECS, etc) go through
func CommonSimpleBootstrap(tPid *identifier.ProcessIdentifier, customMembers FultonCustomMembers, configSetLoader ConfigSetLoader) (*FultonBootstrap, error) {
	// load the config
	set, err := configSetLoader()
	if err != nil {
		return nil, err
	}
	if set == nil {
		return nil, errors.New("got back nil c7 set")
	}
	cfg := c7.NewC7(*set, tPid.Region, tPid.Stage)

	// Construct the logging OpMonitor
	logOpMonitor := &logmiddleware.OperationMonitor{Logger: customMembers.Logger}

	// Construct the metrics OpMonitor
	sampleReporter := telemetry.SampleReporter{
		SampleBuilder:  telemetry.SampleBuilder{ProcessIdentifier: *tPid},
		SampleObserver: customMembers.SampleObserver,
	}
	metricsOpMonitor := &metricsmiddleware.OperationMonitor{
		SampleReporter: sampleReporter,
		AutoFlush:      customMembers.AutoFlushMetrics,
	}

	opStarter := &operation.Starter{OpMonitors: []operation.OpMonitor{metricsOpMonitor, logOpMonitor}}

	// return
	return &FultonBootstrap{
		C7:                cfg,
		Logger:            customMembers.Logger,
		OperationStarter:  opStarter,
		ProcessIdentifier: tPid,
		SampleReporter:    sampleReporter,
	}, nil
}
