package logger

import (
	"fmt"
	"strconv"
	"strings"

	"github.com/pkg/errors"
)

// CausedError is any error that has a Cause() method that returns a wrapped error
type CausedError interface {
	error
	Cause() error
}

// see docs in github.com/pkg/errors for information about this interface
type stackTracer interface {
	StackTrace() errors.StackTrace
}

type Stack []PrintableStackFrame

type PrintableStackFrame struct {
	Filename string `json:"filename"`
	Method   string `json:"method"`
	Line     int    `json:"lineno"`
}

//
// GetStackChain returns a list of stack traces, one for each layer of err
// For each error, we see if it has a stack trace included and if it does we include it.
// For each error, we then see if the error has a Cause, and if it does, we recurse into that error.
// The expectation is that you will errors.Wrap where the error is first encountered by your code which will include
// the stack trace for that part of your codebase.
// github.com/pkg/errors methods that store stacktraces are: New, Errorf, Wrap, and Wrapf
//
type TraceChain []TraceEntry

func GetStackChain(err error) *TraceChain {
	var traceChain TraceChain
	for err != nil {
		stack := getStack(err)
		traceChain = append(traceChain, buildTrace(err, stack))
		if cs, ok := err.(CausedError); ok {
			err = cs.Cause()
		} else {
			break
		}
	}
	return &traceChain
}

type TraceEntry struct {
	Exception string `json:"exception"`
	Frames    Stack  `json:"stack"`
}

// builds one trace element in trace_chain
func buildTrace(err error, stack Stack) TraceEntry {
	return TraceEntry{
		Frames: stack,
		Exception: err.Error(),
	}
}

// gets Stack from errors that provide one of their own
func getStack(err error) Stack {
	cs, ok := err.(stackTracer)

	if !ok {
		return nil
	}

	stackTrace := cs.StackTrace()
	var stack Stack
	for _, stackFrame := range stackTrace {
		stack = append(stack, stackFrameToPrintableStackFrame(stackFrame))
	}

	return stack
}

// use custom printf verbs to extract information out of the errors.Frame object.  See the source for
// errors.Frame for documentation on these verbs.
func stackFrameToPrintableStackFrame(frame errors.Frame) PrintableStackFrame {
	lineNumber, _ := strconv.Atoi(fmt.Sprintf("%d", frame))
	// nolint: vet
	filename := fmt.Sprintf("%+s", frame)
	return PrintableStackFrame{
		Filename: shortenFilePath(filename),
		// nolint: vet
		Method: fmt.Sprintf("%n", frame),
		Line: lineNumber,
	}
}

//
// Remove un-needed information from the source file path.
//
// Examples:
//   /usr/local/go/src/pkg/runtime/proc.c -> pkg/runtime/proc.c
//
func shortenFilePath(s string) string {
	idx := strings.Index(s, "/src/")
	if idx != -1 {
		return s[idx+5:]
	}
	return s
}

