package manager

import (
	"context"

	"code.justin.tv/extensions/discovery/auth"
	discovery "code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/interfaces"
	"code.justin.tv/extensions/discovery/sliceutils"
	"code.justin.tv/extensions/discovery/twirputils"
)

const (
	defaultPaginationLimit = 10
)

var (
	_ interfaces.DiscoveryManager = &manager{}
)

type manager struct {
	store model.Store
}

// New returns an implementation of the DiscoveryManager
func New(store model.Store) interfaces.DiscoveryManager {
	return &manager{
		store: store,
	}
}

func (m *manager) CreateCategory(ctx context.Context, req *discovery.AddCategoryRequest) (*discovery.CategoryDocument, error) {
	if !m.creds(ctx).CanEditAllCategories() {
		return nil, data.ErrUnauthorized
	}

	err := data.ValidateCategoryUpsert(req.GetCategory(), false)
	if err != nil {
		return nil, err
	}

	category, err := m.store.CreateCategory(req)
	if err != nil {
		return nil, err
	}

	categoryDoc := model.BuildCategoryDocument(category, nil)
	return categoryDoc, nil
}

func (m *manager) UpdateCategory(ctx context.Context, req *discovery.UpdateCategoryRequest) (*discovery.CategoryDocument, error) {
	err := data.ValidateCategoryUpsert(req.GetCategory(), true)
	if err != nil {
		return nil, err
	}

	if !m.creds(ctx).CanEditAllCategories() {
		return nil, data.ErrUnauthorized
	}

	category, err := m.store.UpdateCategory(req)
	if err != nil {
		return nil, err
	}

	categoryDoc := model.BuildCategoryDocument(category, nil)
	return categoryDoc, nil
}

func (m *manager) GetCategory(ctx context.Context, req *discovery.GetCategoryRequest) (*discovery.CategoryDocument, error) {
	category, translation, err := m.store.GetCategory(req)
	if err != nil {
		return nil, err
	}

	// all categories can be fetched by id (even visible=false). if they are deleted, we should hide them.
	if !category.Deleted || m.creds(ctx).CanEditAllCategories() {
		categoryDoc := model.BuildCategoryDocument(category, translation)
		return categoryDoc, nil
	}

	return nil, data.ErrCategoryNotFound
}

func (m *manager) GetCategories(ctx context.Context, req *discovery.GetCategoriesRequest) (*discovery.CategoriesDocument, error) {
	if !m.creds(ctx).CanEditAllCategories() {
		req.IncludeHidden = false
		req.IncludeDeleted = false
	}

	if req.GetLimit() == 0 {
		req.Limit = defaultPaginationLimit
	}

	categories, translations, totalCategories, err := m.store.GetCategories(req)
	if err != nil {
		return nil, err
	}

	categoriesDoc, err := model.BuildCategoriesDocument(categories, translations, totalCategories)
	if err != nil {
		return nil, err
	}

	return categoriesDoc, nil
}

func (m *manager) EditCategoryTranslation(ctx context.Context, req *discovery.EditCategoryTranslationRequest) (*discovery.CategoryDocument, error) {
	if !m.creds(ctx).CanEditAllCategories() {
		return nil, data.ErrUnauthorized
	}

	translation, category, err := m.store.EditCategoryTranslation(req)
	if err != nil {
		return nil, err
	}

	categoryDoc := model.BuildCategoryDocument(category, translation)
	return categoryDoc, nil
}

func (m *manager) OrderCategories(ctx context.Context, req *discovery.OrderCategoriesRequest) ([]*discovery.CategoryDocument, error) {
	if !m.creds(ctx).CanEditAllCategories() {
		return nil, data.ErrUnauthorized
	}

	categories, err := m.store.OrderCategories(req)
	if err != nil {
		return nil, err
	}

	count := len(categories)

	controllers := make([]*discovery.CategoryDocument, count)
	for i := 0; i < count; i++ {
		categoryDoc := model.BuildCategoryDocument(categories[i], nil)
		controllers[i] = categoryDoc
	}
	return controllers, nil
}

func (m *manager) OrderCategory(ctx context.Context, req *discovery.OrderCategoryRequest) (*discovery.CategoryDocument, error) {
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}

	category, err := m.store.OrderCategory(req)
	if err != nil {
		return nil, err
	}

	categoryDoc := model.BuildCategoryDocument(category, nil)
	return categoryDoc, nil
}

func (m *manager) AddExtensionToDeveloperCategory(ctx context.Context, req *discovery.AddExtensionToCategoryRequest) (*discovery.ExtensionCategoryMembershipDocument, error) {
	err := data.ValidateAddExtensionToCategory(req)
	if err != nil {
		return nil, err
	}

	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}

	cats := []string{req.GetCategoryID()}
	discoveryData, err := m.store.UpdateDiscoveryData(&discovery.UpdateExtensionVersionDiscoveryDataRequest{
		UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
			ExtensionID: req.GetExtensionID(),
			Version:     req.GetVersionID().GetValue(),
			Categories: &discovery.OptionalStringList{
				Values: cats,
			},
		},
	})
	if err != nil {
		return nil, err
	}

	newCats := discoveryData.Categories
	// This should only be possible of the database is broken, but better to have a redundant check than NPE
	if len(newCats) != 1 {
		return nil, data.ErrExtensionNotInCategory
	}

	return &discovery.ExtensionCategoryMembershipDocument{
		CategoryID:  discoveryData.Categories[0],
		ExtensionID: discoveryData.ExtensionID,
	}, nil
}

func (m *manager) AddExtensionToCuratedCategory(ctx context.Context, req *discovery.AddExtensionToCategoryRequest) (*discovery.ExtensionCategoryMembershipDocument, error) {
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}

	cat, _, err := m.store.GetCategory(&discovery.GetCategoryRequest{
		CategoryID: req.CategoryID,
	})
	if err != nil {
		return nil, err
	}

	if cat.Type != model.CategoryTypeCurated || req == nil {
		return nil, data.ErrInvalidCategoryUpdate
	}

	relationDoc, err := m.store.AddExtensionToCategory(req)
	if err != nil {
		return nil, err
	}
	startTime, err := twirputils.WrapTimePointer(relationDoc.StartTime)
	if err != nil {
		return nil, err
	}

	return &discovery.ExtensionCategoryMembershipDocument{
		CategoryID:  relationDoc.CategoryID,
		ExtensionID: relationDoc.ExtensionID,
		StartTime:   startTime,
		Order:       twirputils.WrapFloat(relationDoc.Order),
	}, nil
}

func (m *manager) GetExtensionsForCategory(ctx context.Context, req *discovery.GetCategoryExtensionsRequest) (*discovery.CategoryExtensionsDocument, error) {
	if req.GetLimit() == 0 {
		req.Limit = defaultPaginationLimit
	}

	// Verify category exists
	// TODO: Maybe remove check and just ensure stores handle it?
	// (check copied from EMS though so can't do it unilaterally)
	_, err := m.GetCategory(ctx, &discovery.GetCategoryRequest{
		CategoryID: req.GetCategoryID(),
	})
	if err != nil {
		return nil, err
	}

	// TODO EMS calls search for developer cat exts, whitelisted, and new released
	// But not curated

	extensions, count, err := m.store.GetExtensionsForCategory(req)
	if err != nil {
		return nil, err
	}

	ids := make([]string, len(extensions))
	for i, extension := range extensions {
		ids[i] = extension.ExtensionID
	}

	return &discovery.CategoryExtensionsDocument{
		IDs:        ids,
		TotalCount: count,
	}, nil
}

func (m *manager) RemoveExtensionFromCategory(ctx context.Context, req *discovery.RemoveExtensionFromCategoryRequest) (*discovery.ExtensionCategoryMembershipDocument, error) {
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}

	cat, _, err := m.store.GetCategory(&discovery.GetCategoryRequest{
		CategoryID: req.CategoryID,
	})
	if err != nil {
		return nil, err
	}

	if cat.Type != model.CategoryTypeCurated {
		return nil, data.ErrInvalidCategoryUpdate
	}

	relationModel, err := m.store.RemoveExtensionFromCategory(req)
	if err != nil {
		return nil, err
	}

	return &discovery.ExtensionCategoryMembershipDocument{
		CategoryID:  relationModel.CategoryID,
		ExtensionID: relationModel.ExtensionID,
	}, nil
}

func (m *manager) DeleteCategory(ctx context.Context, req *discovery.DeleteCategoryRequest) (*discovery.DeleteCategoryResponse, error) {
	if !m.creds(ctx).CanEditAllCategories() {
		return nil, data.ErrUnauthorized
	}

	cat, _, err := m.store.GetCategory(&discovery.GetCategoryRequest{
		CategoryID: req.GetCategoryID(),
	})
	if err != nil {
		return nil, err
	}

	//
	// To delete a developer category, you must have first:
	// 1.  the category must have no extensions in it
	// 2.  not visible
	// 3.  read-only so people can't be adding extensions to the category at the same time
	//
	softDelete := false
	if cat.Type == model.CategoryTypeDeveloper {
		// 1. needs to be read-only
		if !cat.Readonly {
			return nil, data.ErrNotReadOnly
		}
		// 2. needs to be invisible
		if cat.Visible {
			return nil, data.ErrCategoryVisible
		}
		// 3. verify no extensions are in this category
		exts, err := m.GetExtensionsForCategory(ctx, &discovery.GetCategoryExtensionsRequest{
			Limit:      1, // use 1 because if it's 0 then the default value of 25 gets used
			Offset:     0,
			CategoryID: req.GetCategoryID(),
		})
		if err != nil {
			return nil, err
		}

		if exts.GetTotalCount() > 0 {
			return nil, data.ErrNonEmptyCategory
		}
		// 4. soft-delete
		softDelete = true
		cat.Deleted = true
	}

	err = m.store.DeleteCategory(&discovery.DeleteCategoryRequest{
		CategoryID: req.GetCategoryID(),
		SoftDelete: softDelete,
	})
	if err != nil {
		return nil, err
	}

	// TODO don't love reloading here - can avoid?
	cat, _, err = m.store.GetCategory(&discovery.GetCategoryRequest{
		CategoryID: req.GetCategoryID(),
	})
	if err != nil {
		if err == data.ErrNotFound {
			return &discovery.DeleteCategoryResponse{}, nil
		}
		return nil, err
	}

	catDoc := model.BuildCategoryDocument(cat, nil)
	return &discovery.DeleteCategoryResponse{
		Category: catDoc,
	}, nil
}

func (m *manager) CreateFeaturedSchedule(ctx context.Context, req *discovery.AddFeaturedScheduleRequest) (*discovery.FeaturedScheduleDocument, error) {
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}

	schedule, err := m.store.CreateFeaturedSchedule(req)
	if err != nil {
		return nil, err
	}

	resp, err := model.BuildFeaturedScheduleDocument(schedule, nil, nil)
	if err != nil {
		return nil, err
	}

	return resp, nil
}

func (m *manager) GetFeaturedSchedules(ctx context.Context, req *discovery.GetFeaturedSchedulesRequest) (*discovery.FeaturedSchedulesDocument, error) {
	if req.GetLimit() == 0 {
		req.Limit = defaultPaginationLimit
	}

	schedules, totalSchedules, err := m.store.GetFeaturedSchedules(req)
	if err != nil {
		return nil, err
	}
	count := len(schedules)
	scheduleIDs := make([]string, count)
	for i := 0; i < count; i++ {
		scheduleIDs[i] = schedules[i].ID
	}

	carousels, entries, err := m.getCarouselsAndEntriesByScheduleIDs(ctx, scheduleIDs)
	if err != nil {
		return nil, err
	}

	resp, err := model.BuildFeaturedSchedulesDocument(schedules, totalSchedules, carousels, entries)
	if err != nil {
		return nil, err
	}

	return resp, nil
}

func (m *manager) getCarouselsAndEntriesByScheduleIDs(ctx context.Context, scheduleIDs []string) ([]*model.FeaturedCarousel, [][]*model.FeaturedCarouselEntry, error) {
	carousels, err := m.store.GetCarouselsByScheduleIDs(scheduleIDs)
	if err != nil {
		return nil, nil, err
	}

	carouselCount := len(carousels)
	carouselIDs := make([]string, carouselCount)
	for i := 0; i < carouselCount; i++ {
		carouselIDs[i] = carousels[i].ID
	}
	entries, err := m.store.GetCarouselEntriesByCarouselIDs(carouselIDs)
	if err != nil {
		return nil, nil, err
	}

	return carousels, entries, nil
}

func (m *manager) getEntriesByCarouselIDs(ctx context.Context, carouselIDs []string) ([][]*model.FeaturedCarouselEntry, error) {
	entries, err := m.store.GetCarouselEntriesByCarouselIDs(carouselIDs)
	if err != nil {
		return nil, err
	}

	return entries, nil
}

func (m *manager) getEntriesByCarouselID(ctx context.Context, id string) ([]*model.FeaturedCarouselEntry, error) {
	entries, err := m.getEntriesByCarouselIDs(ctx, []string{id})
	if err != nil {
		return nil, err
	}

	if len(entries) < 1 {
		return nil, data.ErrNotFound
	}

	return entries[0], nil
}

func (m *manager) GetFeaturedSchedule(ctx context.Context, req *discovery.GetFeaturedScheduleRequest) (*discovery.FeaturedScheduleDocument, error) {
	schedule, err := m.store.GetFeaturedSchedule(req)
	if err != nil {
		return nil, err
	}

	carousels, entries, err := m.getCarouselsAndEntriesByScheduleIDs(ctx, []string{req.ScheduleID})
	if err != nil {
		return nil, err
	}

	// Ensure it can be indexed into
	if len(carousels) != 1 || len(entries) != 1 {
		return nil, data.ErrNotFound
	}

	resp, err := model.BuildFeaturedScheduleDocument(schedule, carousels[0], entries[0])
	if err != nil {
		return nil, err
	}
	return resp, nil
}

func (m *manager) DeleteFeaturedSchedule(ctx context.Context, req *discovery.DeleteFeaturedScheduleRequest) (*discovery.FeaturedScheduleDocument, error) {
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}
	schedule, err := m.store.DeleteFeaturedSchedule(req)
	if err != nil {
		return nil, err
	}
	return model.BuildFeaturedScheduleDocument(schedule, nil, nil)
}

func (m *manager) UpdateFeaturedSchedule(ctx context.Context, req *discovery.UpdateFeaturedScheduleRequest) (*discovery.FeaturedScheduleDocument, error) {
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}

	schedule, err := m.store.UpdateFeaturedSchedule(req)
	if err != nil {
		return nil, err
	}

	carousels, entries, err := m.getCarouselsAndEntriesByScheduleIDs(ctx, []string{req.ScheduleID})
	if err != nil {
		return nil, err
	}

	// Ensure it can be indexed into
	if len(carousels) != 1 || len(entries) != 1 {
		return nil, data.ErrNotFound
	}

	resp, err := model.BuildFeaturedScheduleDocument(schedule, carousels[0], entries[0])
	if err != nil {
		return nil, err
	}
	return resp, nil
}

func (m *manager) CreateFeaturedCarousel(ctx context.Context, req *discovery.AddFeaturedCarouselRequest) (*discovery.FeaturedCarouselDocument, error) {
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}

	carousel, err := m.store.CreateFeaturedCarousel(req)
	if err != nil {
		return nil, err
	}

	entries, err := m.getEntriesByCarouselID(ctx, carousel.ID)
	if err != nil {
		return nil, err
	}

	return model.BuildFeaturedCarouselDocument(carousel, entries, int32(len(entries)))
}

func (m *manager) GetFeaturedCarousel(ctx context.Context, req *discovery.GetFeaturedCarouselRequest) (*discovery.FeaturedCarouselDocument, error) {
	carousel, err := m.store.GetFeaturedCarousel(req)
	if err != nil {
		return nil, err
	}
	entries, err := m.getEntriesByCarouselID(ctx, carousel.ID)
	if err != nil {
		return nil, err
	}

	return model.BuildFeaturedCarouselDocument(carousel, entries, int32(len(entries)))
}

func (m *manager) GetFeaturedCarousels(ctx context.Context, req *discovery.GetFeaturedCarouselsRequest) (*discovery.FeaturedCarouselsDocument, error) {
	if !m.creds(ctx).CanCurateAllCategories() {
		// this is intended for admins only
		return nil, data.ErrUnauthorized
	}

	if req.GetLimit() == 0 {
		req.Limit = defaultPaginationLimit
	}

	carousels, totalCarousels, err := m.store.GetFeaturedCarousels(req)
	if err != nil {
		return nil, err
	}

	carouselCount := len(carousels)
	carouselIDs := make([]string, carouselCount)
	for i := 0; i < carouselCount; i++ {
		carouselIDs[i] = carousels[i].ID
	}

	entries, err := m.getEntriesByCarouselIDs(ctx, carouselIDs)
	if err != nil {
		return nil, err
	}
	// Entries should have one per carousel ID passed in even if a given ID had no associated carousel entries
	if len(entries) != carouselCount {
		return nil, data.ErrNotFound
	}

	return model.BuildFeaturedCarouselsDocument(carousels, entries, totalCarousels)
}

func (m *manager) DeleteFeaturedCarousel(ctx context.Context, req *discovery.DeleteFeaturedCarouselRequest) (*discovery.FeaturedCarouselDocument, error) {
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}
	carousel, err := m.store.DeleteFeaturedCarousel(req)
	if err != nil {
		return nil, err
	}
	return model.BuildFeaturedCarouselDocument(carousel, nil, 0)
}

func (m *manager) UpdateFeaturedCarousel(ctx context.Context, req *discovery.UpdateFeaturedCarouselRequest) (*discovery.FeaturedCarouselDocument, error) {
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}
	carousel, err := m.store.UpdateFeaturedCarousel(req)
	if err != nil {
		return nil, err
	}
	entries, err := m.getEntriesByCarouselID(ctx, carousel.ID)
	if err != nil {
		return nil, err
	}

	return model.BuildFeaturedCarouselDocument(carousel, entries, int32(len(entries)))
}

func (m *manager) OrderFeaturedCarousel(ctx context.Context, req *discovery.OrderFeaturedCarouselRequest) (*discovery.FeaturedCarouselDocument, error) {
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}

	carousel, err := m.store.OrderFeaturedCarousel(req)
	if err != nil {
		return nil, err
	}

	entries, err := m.getEntriesByCarouselID(ctx, carousel.ID)
	if err != nil {
		return nil, err
	}

	return model.BuildFeaturedCarouselDocument(carousel, entries, int32(len(entries)))
}

func (m *manager) CreateFeaturedCarouselEntry(ctx context.Context, req *discovery.AddFeaturedCarouselEntryRequest) (*discovery.FeaturedCarouselEntryDocument, error) {
	err := data.ValidateAddFeaturedCarouselEntry(req)
	if err != nil {
		return nil, err
	}

	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}

	entry, err := m.store.CreateFeaturedCarouselEntry(req)
	if err != nil {
		return nil, err
	}

	return model.BuildFeaturedCarouselEntryDocument(entry)
}

func (m *manager) GetFeaturedCarouselEntry(ctx context.Context, req *discovery.GetFeaturedCarouselEntryRequest) (*discovery.FeaturedCarouselEntryDocument, error) {
	entry, err := m.store.GetFeaturedCarouselEntry(req)
	if err != nil {
		return nil, err
	}

	return model.BuildFeaturedCarouselEntryDocument(entry)
}

func (m *manager) GetFeaturedCarouselEntries(ctx context.Context, req *discovery.GetFeaturedCarouselEntriesRequest) (*discovery.FeaturedCarouselEntriesDocument, error) {
	if !m.creds(ctx).CanCurateAllCategories() {
		// this is intended for admins only
		return nil, data.ErrUnauthorized
	}

	if req.GetLimit() == 0 {
		req.Limit = defaultPaginationLimit
	}

	entries, totalEntries, err := m.store.GetFeaturedCarouselEntries(req)
	if err != nil {
		return nil, err
	}

	return model.BuildFeaturedCarouselEntriesDocument(entries, totalEntries)
}

func (m *manager) DeleteFeaturedCarouselEntry(ctx context.Context, req *discovery.DeleteFeaturedCarouselEntryRequest) (*discovery.FeaturedCarouselEntryDocument, error) {
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}

	entry, err := m.store.DeleteFeaturedCarouselEntry(req)
	if err != nil {
		return nil, err
	}

	return model.BuildFeaturedCarouselEntryDocument(entry)
}

func (m *manager) UpdateFeaturedCarouselEntry(ctx context.Context, req *discovery.UpdateFeaturedCarouselEntryRequest) (*discovery.FeaturedCarouselEntryDocument, error) {
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}

	entry, err := m.store.UpdateFeaturedCarouselEntry(req)
	if err != nil {
		return nil, err
	}

	return model.BuildFeaturedCarouselEntryDocument(entry)
}

func (m *manager) PutExtensionVersionDiscoveryData(ctx context.Context, req *discovery.PutExtensionVersionDiscoveryDataRequest) (*discovery.ExtensionVersionDiscoveryDocument, error) {
	// TODO is this the correct permissions check?
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}

	discoData, err := m.store.PutDiscoveryData(req)
	if err != nil {
		return nil, err
	}

	return model.BuildExtensionVersionDiscoveryDocument(discoData)
}

func (m *manager) AddGameToExtension(ctx context.Context, req *discovery.AddGameToExtensionRequest) (*discovery.AddGameToExtensionResponse, error) {
	// Uses UpdateExtensionVersionDiscoveryDataRequest, updating the game field
	resp, err := m.UpdateExtensionVersionDiscoveryData(ctx, &discovery.UpdateExtensionVersionDiscoveryDataRequest{
		UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
			ExtensionID: req.GetExtensionID(),
			Version:     req.GetVersionID(),
			Games: &discovery.OptionalIntList{
				Values: []int64{int64(req.GetGameID())},
			},
		},
	})
	if err != nil {
		return nil, err
	}

	games := resp.GetDiscoveryData().GetGames()

	// This should only happen if something is very wrong with the DB layer, but better to double check than panic
	if len(games) != 1 {
		return nil, data.ErrLogicViolation
	}

	return &discovery.AddGameToExtensionResponse{
		ExtensionGameMembership: &discovery.ExtensionGameMembershipDocument{
			ExtensionID: resp.GetExtensionID(),
			GameID:      int32(games[0]),
		},
	}, nil
}

func (m *manager) DeleteGameFromExtension(ctx context.Context, req *discovery.DeleteGameFromExtensionRequest) (*discovery.DeleteGameFromExtensionResponse, error) {
	// TODO is this the correct permissions check?
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}

	ext, err := m.GetExtensionVersionDiscoveryData(ctx, &discovery.GetExtensionVersionDiscoveryDataRequest{
		ExtensionID: req.GetExtensionID(),
		Version:     req.GetVersionID(),
	})
	if err != nil {
		return nil, err
	}

	games := ext.GetDiscoveryData().GetGames()
	i, found := sliceutils.Int64Find(int64(req.GetGameID()), games)

	if !found {
		return nil, data.ErrGameNotFound
	}

	games = sliceutils.Int64RemoveIndex(i, games)
	// Uses UpdateExtensionVersionDiscoveryDataRequest, updating the game field
	resp, err := m.UpdateExtensionVersionDiscoveryData(ctx, &discovery.UpdateExtensionVersionDiscoveryDataRequest{
		UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
			ExtensionID: req.GetExtensionID(),
			Version:     req.GetVersionID(),
			Games: &discovery.OptionalIntList{
				Values: games,
			},
		},
	})
	if err != nil {
		return nil, err
	}

	return &discovery.DeleteGameFromExtensionResponse{
		ExtensionGameMembership: &discovery.ExtensionGameMembershipDocument{
			ExtensionID: resp.GetExtensionID(),
		},
	}, nil
}

func (m *manager) UpdateExtensionVersionDiscoveryData(ctx context.Context, req *discovery.UpdateExtensionVersionDiscoveryDataRequest) (*discovery.ExtensionVersionDiscoveryDocument, error) {
	// TODO is this the correct permissions check?
	if !m.creds(ctx).CanCurateAllCategories() {
		return nil, data.ErrUnauthorized
	}

	discoData, err := m.store.UpdateDiscoveryData(req)
	if err != nil {
		return nil, err
	}

	return model.BuildExtensionVersionDiscoveryDocument(discoData)
}

func (m *manager) GetExtensionVersionDiscoveryData(ctx context.Context, req *discovery.GetExtensionVersionDiscoveryDataRequest) (*discovery.ExtensionVersionDiscoveryDocument, error) {
	var discoData *model.ExtensionDiscoveryData
	var err error

	// TODO this is almost certainly not the correct permissions check
	if m.creds(ctx).CanCurateAllCategories() {
		discoData, err = m.store.AdminGetDiscoveryData(req)
	} else {
		discoData, err = m.store.GetDiscoveryData(req)
	}
	if err != nil {
		return nil, err
	}

	return model.BuildExtensionVersionDiscoveryDocument(discoData)
}

func (m *manager) DeleteExtensionVersionDiscoveryData(ctx context.Context, req *discovery.DeleteExtensionVersionDiscoveryDataRequest) error {
	if !m.creds(ctx).CanCurateAllCategories() {
		return data.ErrUnauthorized
	}

	return m.store.SoftDeleteDiscoveryVersionData(req)
}

func (m *manager) HardDeleteExtensionDiscoveryData(ctx context.Context, req *discovery.HardDeleteExtensionDiscoveryDataRequest) error {
	if !m.creds(ctx).CanCurateAllCategories() {
		return data.ErrUnauthorized
	}

	return m.store.HardDeleteDiscoveryData(req)
}

func (m *manager) IsResetEnabled() bool {
	return m.store.IsResetEnabled()
}

func (m *manager) ResetAllData() error {
	if m.IsResetEnabled() {
		return m.store.ResetAllData()
	}

	return data.ErrUnavailable
}

func (m *manager) creds(ctx context.Context) auth.Credentials {
	return auth.Load(ctx)
}
