package manager

import (
	"context"
	"fmt"
	"testing"

	"code.justin.tv/extensions/discovery/auth"
	"code.justin.tv/extensions/discovery/cmd/discovery/rpc"
	"code.justin.tv/extensions/discovery/data"
	"code.justin.tv/extensions/discovery/data/model"
	"code.justin.tv/extensions/discovery/data/model/memory"
	"code.justin.tv/extensions/discovery/data/model/modelfakes"
	"code.justin.tv/extensions/discovery/golibs/uuid"
	"code.justin.tv/extensions/discovery/interfaces"
	"code.justin.tv/extensions/discovery/twirputils"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

var (
	// Go has the worst linting rules and I hate it
	errTestError = fmt.Errorf("testing error")
)

func TestManager(t *testing.T) {
	setup := func(creds auth.Credentials) (context.Context, interfaces.DiscoveryManager) {
		store := memory.New(uuid.NewSource())
		manager := New(store)
		return auth.Store(context.Background(), creds), manager
	}

	t.Run("CreateCategory", func(t *testing.T) {
		t.Run("requestor cannot edit categories", func(t *testing.T) {
			ctx, m := setup(&auth.FakeCredentials{})
			resp, err := m.CreateCategory(ctx, validAddCategoryRequest())
			assert.Nil(t, resp)
			assert.Error(t, err)
			assert.EqualError(t, err, data.ErrUnauthorized.Error())
		})

		t.Run("requestor can edit categories", func(t *testing.T) {
			ctx, m := setup(&auth.FakeCredentials{Role: auth.Categorizer})
			resp, err := m.CreateCategory(ctx, validAddCategoryRequest())
			assert.NotNil(t, resp)
			assert.NoError(t, err)
		})

		t.Run("create store failure", func(t *testing.T) {
			store := &modelfakes.FakeStore{}
			store.CreateCategoryStub = func(*discovery.AddCategoryRequest) (*model.Category, error) {
				return nil, errTestError
			}
			m := New(store)
			ctx := auth.Store(context.Background(), &auth.FakeCredentials{Role: auth.Categorizer})

			resp, err := m.CreateCategory(ctx, validAddCategoryRequest())

			assert.Nil(t, resp)
			assert.Equal(t, errTestError, err)
			assert.Equal(t, 1, store.CreateCategoryCallCount())
		})
	})

	t.Run("UpdateCategory", func(t *testing.T) {
		t.Run("invalid input", func(t *testing.T) {
			ctx, m := setup(&auth.FakeCredentials{Role: auth.Categorizer})
			update := &discovery.UpdateCategoryRequest{
				Category: &discovery.CategoryUpsert{
					Type: twirputils.WrapString("banana"),
				},
			}

			resp, err := m.UpdateCategory(ctx, update)
			assert.Nil(t, resp)
			assert.Error(t, err)
			assert.EqualError(t, err, data.ErrUnknownCategoryType.Error())
		})

		t.Run("requestor cannot edit categories", func(t *testing.T) {
			ctx, m := setup(&auth.FakeCredentials{})
			cat := createCategoryWithNoErrors(t, m, validAddCategoryRequest())
			update := validUpdateCategoryRequest(cat.GetID())
			update.Category.Order = twirputils.WrapFloat(100)

			resp, err := m.UpdateCategory(ctx, update)
			assert.Nil(t, resp)
			assert.Error(t, err)
			assert.EqualError(t, err, data.ErrUnauthorized.Error())
		})

		t.Run("requestor can edit categories", func(t *testing.T) {
			ctx, m := setup(&auth.FakeCredentials{Role: auth.Categorizer})
			cat := createCategoryWithNoErrors(t, m, validAddCategoryRequest())
			update := validUpdateCategoryRequest(cat.GetID())
			update.Category.Order = twirputils.WrapFloat(100)

			resp, err := m.UpdateCategory(ctx, update)
			assert.NotNil(t, resp)
			assert.NoError(t, err)
		})

		t.Run("update store failure", func(t *testing.T) {
			store := &modelfakes.FakeStore{}
			store.UpdateCategoryStub = func(*discovery.UpdateCategoryRequest) (*model.Category, error) {
				return nil, errTestError
			}
			m := New(store)
			ctx := auth.Store(context.Background(), &auth.FakeCredentials{Role: auth.Categorizer})
			update := validUpdateCategoryRequest("id")
			update.Category.Order = twirputils.WrapFloat(100)

			resp, err := m.UpdateCategory(ctx, update)

			assert.Nil(t, resp)
			assert.Equal(t, errTestError, err)
			assert.Equal(t, 1, store.UpdateCategoryCallCount())
		})
	})

	t.Run("GetCategory", func(t *testing.T) {
		t.Run("category exists", func(t *testing.T) {
			ctx, m := setup(&auth.FakeCredentials{Role: auth.Categorizer})
			cat := createCategoryWithNoErrors(t, m, validAddCategoryRequest())
			resp, err := m.GetCategory(ctx, &discovery.GetCategoryRequest{
				CategoryID: cat.GetID(),
				Language:   model.DefaultCategoryLanguageCode,
			})
			assert.NoError(t, err)
			assert.NotNil(t, resp)
		})

		t.Run("get store failure", func(t *testing.T) {
			store := &modelfakes.FakeStore{}
			store.GetCategoryStub = func(*discovery.GetCategoryRequest) (*model.Category, *model.CategoryTranslation, error) {
				return nil, nil, errTestError
			}
			m := New(store)

			resp, err := m.GetCategory(context.Background(), &discovery.GetCategoryRequest{})

			assert.Nil(t, resp)
			assert.Equal(t, errTestError, err)
			assert.Equal(t, 1, store.GetCategoryCallCount())
		})

		t.Run("category does not exist", func(t *testing.T) {
			ctx, m := setup(&auth.FakeCredentials{Role: auth.Categorizer})
			resp, err := m.GetCategory(ctx, &discovery.GetCategoryRequest{
				CategoryID: "id",
				Language:   model.DefaultCategoryLanguageCode,
			})
			assert.Error(t, err)
			assert.Nil(t, resp)
			assert.Equal(t, data.ErrNotFound, err)
		})

		t.Run("category exists but deleted", func(t *testing.T) {
			ctx, m := setup(&auth.FakeCredentials{})
			cat := createAndSoftDeleteCategoryWithNoErrors(t, m, validAddCategoryRequest())

			resp, err := m.GetCategory(ctx, &discovery.GetCategoryRequest{
				CategoryID: cat.GetID(),
				Language:   model.DefaultCategoryLanguageCode,
			})
			assert.Error(t, err)
			assert.Nil(t, resp)
			assert.Equal(t, data.ErrCategoryNotFound, err)
		})

	})

	t.Run("GetCategories", func(t *testing.T) {
		ctx, m := setup(&auth.FakeCredentials{})

		cat1 := validAddCategoryRequest()
		cat1.Category.Order = twirputils.WrapFloat(1)
		cat1.Category.Visible = twirputils.WrapBool(false)
		cat2 := validAddCategoryRequest()
		cat2.Category.Order = twirputils.WrapFloat(2)
		cat3 := validAddCategoryRequest()
		cat3.Category.Order = twirputils.WrapFloat(3)

		createCategoryWithNoErrors(t, m, cat1)
		createCategoryWithNoErrors(t, m, cat2)
		createCategoryWithNoErrors(t, m, cat3)

		t.Run("requestor cannot edit categories", func(t *testing.T) {
			t.Run("does not return hidden entries", func(t *testing.T) {
				req := &discovery.GetCategoriesRequest{
					Type:           model.CategoryTypeDeveloper,
					Offset:         0,
					Limit:          10,
					IncludeDeleted: true,
					IncludeHidden:  true,
					Language:       model.DefaultCategoryLanguageCode,
				}
				resp, err := m.GetCategories(ctx, req)
				assert.NoError(t, err)
				assert.Equal(t, int32(2), resp.Count)
				assert.Equal(t, 2, len(resp.Categories))
			})

			t.Run("does not return deleted entries", func(t *testing.T) {
				// TODO
			})
		})

		t.Run("requestor can edit categories", func(t *testing.T) {
			ctx2 := setTempCreds(m, &auth.FakeCredentials{Role: auth.Categorizer})
			t.Run("returns hidden", func(t *testing.T) {
				req := &discovery.GetCategoriesRequest{
					Type:           model.CategoryTypeDeveloper,
					Offset:         0,
					Limit:          10,
					IncludeDeleted: true,
					IncludeHidden:  true,
					Language:       model.DefaultCategoryLanguageCode,
				}
				resp, err := m.GetCategories(ctx2, req)
				assert.NoError(t, err)
				assert.Equal(t, int32(3), resp.Count)
				assert.Equal(t, 3, len(resp.Categories))
			})

			t.Run("returns deleted", func(t *testing.T) {
				// TODO
			})
		})

		t.Run("get store failure", func(t *testing.T) {
			store := &modelfakes.FakeStore{}
			store.GetCategoriesStub = func(*discovery.GetCategoriesRequest) ([]*model.Category, []*model.CategoryTranslation, int32, error) {
				return nil, nil, 0, errTestError
			}
			m := New(store)

			resp, err := m.GetCategories(context.Background(), &discovery.GetCategoriesRequest{})

			assert.Nil(t, resp)
			assert.Equal(t, errTestError, err)
			assert.Equal(t, 1, store.GetCategoriesCallCount())
		})

		t.Run("translation count mismatch", func(t *testing.T) {
			store := &modelfakes.FakeStore{}
			store.GetCategoriesStub = func(*discovery.GetCategoriesRequest) ([]*model.Category, []*model.CategoryTranslation, int32, error) {
				return []*model.Category{}, []*model.CategoryTranslation{&model.CategoryTranslation{}}, 0, nil
			}
			m := New(store)

			resp, err := m.GetCategories(context.Background(), &discovery.GetCategoriesRequest{})

			assert.Nil(t, resp)
			assert.Equal(t, data.ErrTranslationCountMismatch, err)
			assert.Equal(t, 1, store.GetCategoriesCallCount())
		})
	})

	t.Run("EditCategoryTranslation", func(t *testing.T) {
		t.Run("requestor cannot edit categories", func(t *testing.T) {
			ctx, m := setup(&auth.FakeCredentials{})
			cat := createCategoryWithNoErrors(t, m, validAddCategoryRequest())

			resp, err := m.EditCategoryTranslation(ctx, validEditCategoryTranslationRequest(cat.GetID(), "de"))
			assert.Nil(t, resp)
			assert.Error(t, err)
			assert.EqualError(t, err, data.ErrUnauthorized.Error())
		})

		t.Run("requestor can edit categories", func(t *testing.T) {
			ctx, m := setup(&auth.FakeCredentials{Role: auth.Categorizer})
			cat := createCategoryWithNoErrors(t, m, validAddCategoryRequest())

			resp, err := m.EditCategoryTranslation(ctx, validEditCategoryTranslationRequest(cat.GetID(), "de"))
			assert.NotNil(t, resp)
			assert.NoError(t, err)
		})

		t.Run("edit store failure", func(t *testing.T) {
			store := &modelfakes.FakeStore{}
			store.EditCategoryTranslationStub = func(*discovery.EditCategoryTranslationRequest) (*model.CategoryTranslation, *model.Category, error) {
				return nil, nil, errTestError
			}
			m := New(store)
			ctx := auth.Store(context.Background(), &auth.FakeCredentials{Role: auth.Categorizer})

			resp, err := m.EditCategoryTranslation(ctx, &discovery.EditCategoryTranslationRequest{})

			assert.Nil(t, resp)
			assert.Equal(t, errTestError, err)
			assert.Equal(t, 1, store.EditCategoryTranslationCallCount())
		})
	})

	t.Run("IsResetEnabled", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		m := New(store)
		assert.False(t, m.IsResetEnabled())
		store.EnableDataReset()
		assert.True(t, m.IsResetEnabled())
	})

	t.Run("ResetAllData", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		m := New(store)
		assert.Error(t, data.ErrUnavailable, m.ResetAllData())
		store.EnableDataReset()
		assert.Nil(t, m.ResetAllData())
	})

	t.Run("AddExtension", func(t *testing.T) {
		t.Run("CuratedCategory", func(t *testing.T) {
			t.Run("credentials cannot curate extensions", func(t *testing.T) {
				ctx, m := setup(&auth.FakeCredentials{})
				createCategoryWithNoErrors(t, m, validAddCategoryRequest())
				resp, err := m.AddExtensionToCuratedCategory(ctx, &discovery.AddExtensionToCategoryRequest{
					CategoryID: "123",
				})
				assert.Nil(t, resp)
				assert.Equal(t, data.ErrUnauthorized, err)
			})

			t.Run("success", func(t *testing.T) {
				ctx, m := setup(&auth.FakeCredentials{
					Role: auth.Categorizer,
				})
				req := validAddCategoryRequest()
				req.Category.Type = twirputils.WrapString(model.CategoryTypeCurated)
				cat := createCategoryWithNoErrors(t, m, req)

				resp, err := m.AddExtensionToCuratedCategory(ctx, &discovery.AddExtensionToCategoryRequest{
					CategoryID:  cat.GetID(),
					ExtensionID: "ext",
					Language:    "de",
				})
				assert.NotNil(t, resp)
				assert.NoError(t, err)
			})

			t.Run("get store failure", func(t *testing.T) {
				store := &modelfakes.FakeStore{}
				store.GetCategoryStub = func(*discovery.GetCategoryRequest) (*model.Category, *model.CategoryTranslation, error) {
					return nil, nil, errTestError
				}
				m := New(store)
				ctx := auth.Store(context.Background(), &auth.FakeCredentials{Role: auth.Categorizer})

				resp, err := m.AddExtensionToCuratedCategory(ctx, &discovery.AddExtensionToCategoryRequest{
					CategoryID: "123",
				})

				assert.Nil(t, resp)
				assert.Equal(t, errTestError, err)
				assert.Equal(t, 1, store.GetCategoryCallCount())
				assert.Equal(t, 0, store.AddExtensionToCategoryCallCount())
			})

			t.Run("add store failure", func(t *testing.T) {
				store := &modelfakes.FakeStore{}
				store.GetCategoryStub = func(*discovery.GetCategoryRequest) (*model.Category, *model.CategoryTranslation, error) {
					return &model.Category{
						Type: model.CategoryTypeCurated,
					}, &model.CategoryTranslation{}, nil
				}
				store.AddExtensionToCategoryStub = func(*discovery.AddExtensionToCategoryRequest) (*model.CuratedCategoryExtension, error) {
					return nil, errTestError
				}
				m := New(store)
				ctx := auth.Store(context.Background(), &auth.FakeCredentials{Role: auth.Categorizer})

				resp, err := m.AddExtensionToCuratedCategory(ctx, &discovery.AddExtensionToCategoryRequest{
					CategoryID: "123",
				})

				assert.Nil(t, resp)
				assert.Equal(t, errTestError, err)
				assert.Equal(t, 1, store.GetCategoryCallCount())
				assert.Equal(t, 1, store.AddExtensionToCategoryCallCount())
			})

			t.Run("not curated category type", func(t *testing.T) {
				store := &modelfakes.FakeStore{}
				store.GetCategoryStub = func(*discovery.GetCategoryRequest) (*model.Category, *model.CategoryTranslation, error) {
					return &model.Category{
						Type: model.CategoryTypePseudo,
					}, &model.CategoryTranslation{}, nil
				}
				m := New(store)
				ctx := auth.Store(context.Background(), &auth.FakeCredentials{Role: auth.Categorizer})

				resp, err := m.AddExtensionToCuratedCategory(ctx, &discovery.AddExtensionToCategoryRequest{
					CategoryID: "123",
				})

				assert.Nil(t, resp)
				assert.Equal(t, data.ErrInvalidCategoryUpdate, err)
			})
		})

		t.Run("DeveloperCategory", func(t *testing.T) {
			t.Run("missingversion", func(t *testing.T) {
				ctx, m := setup(&auth.FakeCredentials{
					Role: auth.Categorizer,
				})
				req := validAddCategoryRequest()
				req.Category.Type = twirputils.WrapString(model.CategoryTypeDeveloper)
				createCategoryWithNoErrors(t, m, req)

				resp, err := m.AddExtensionToDeveloperCategory(ctx, &discovery.AddExtensionToCategoryRequest{
					CategoryID: "123",
				})
				assert.Nil(t, resp)
				assert.Equal(t, data.ErrMissingParameter, err)
			})

			t.Run("valid version", func(t *testing.T) {
				ctx, m := setup(&auth.FakeCredentials{
					Role: auth.Categorizer,
				})
				req := validAddCategoryRequest()
				req.Category.Type = twirputils.WrapString(model.CategoryTypeDeveloper)
				createCategoryWithNoErrors(t, m, req)

				t.Run("extension does not exist", func(t *testing.T) {
					// TODO the extension existence is currently not validated in discovery service
					t.Skip()

					resp, err := m.AddExtensionToDeveloperCategory(ctx, &discovery.AddExtensionToCategoryRequest{
						ExtensionID: "ext-dne",
						CategoryID:  "cat",
						VersionID:   twirputils.WrapString("ext-ver"),
					})
					assert.Nil(t, resp)
					assert.Equal(t, data.ErrNotFound, err)
				})

				t.Run("extension exists", func(t *testing.T) {
					// TODO this is currently not validated in discovery service
					t.Skip()

					/*
						ctx, m := setup(&auth.FakeCredentials{
							Role: auth.Categorizer,
						})
						ext, err := m.PutExtension(ctx, &discovery.PutExtensionRequest{
							Extension: &discovery.Extension{
								ExtensionID: "ext-de",
							},
						})
						require.NoError(t, err)
						assert.NotNil(t, ext)

						resp, err := m.AddExtensionToDeveloperCategory(ctx, &discovery.AddExtensionToCategoryRequest{
							ExtensionID: ext.GetExtensionID(),
							CategoryID:  "cat",
							VersionID:   twirputils.WrapString("ext-ver"),
						})
						assert.NotNil(t, resp)
						assert.Nil(t, err)
					*/
				})
			})
		})

		t.Run("invalid category type", func(t *testing.T) {
			ctx, m := setup(&auth.FakeCredentials{
				Role: auth.Categorizer,
			})
			req := validAddCategoryRequest()
			req.Category.Type = twirputils.WrapString("invalid")
			cat, err := m.CreateCategory(ctx, req)

			assert.Nil(t, cat)
			assert.Equal(t, data.ErrUnknownCategoryType, err)
		})
	})

	t.Run("GetExtensions", func(t *testing.T) {
		t.Run("for curated category", func(t *testing.T) {
			t.Run("extension IDs are fetched from store", func(t *testing.T) {
				ctx, m := setup(&auth.FakeCredentials{
					Role: auth.Categorizer,
				})
				req := validAddCategoryRequest()
				req.Category.Type = twirputils.WrapString(model.CategoryTypeCurated)
				cat := createCategoryWithNoErrors(t, m, req)

				extID := "extension_id_pg_1"
				resp, err := m.AddExtensionToCuratedCategory(ctx, &discovery.AddExtensionToCategoryRequest{
					ExtensionID: extID,
					CategoryID:  cat.GetID(),
					Language:    "de",
				})
				require.NoError(t, err)
				assert.NotNil(t, resp)

				getResp, err := m.GetExtensionsForCategory(ctx, &discovery.GetCategoryExtensionsRequest{
					CategoryID: cat.GetID(),
					Limit:      10,
					Offset:     0,
				})
				assert.Nil(t, err)
				require.NotNil(t, getResp)
				require.Equal(t, 1, len(getResp.GetIDs()))
				assert.Equal(t, extID, getResp.GetIDs()[0])
			})
		})

		t.Run("for developer category", func(t *testing.T) {
			t.Run("extensions are fetched from search", func(t *testing.T) {
				ctx, m := setup(&auth.FakeCredentials{
					Role: auth.Categorizer,
				})
				req := validAddCategoryRequest()
				req.Category.Type = twirputils.WrapString(model.CategoryTypeDeveloper)
				cat := createCategoryWithNoErrors(t, m, req)

				extID := "testExtensionID"
				discoData, err := m.PutExtensionVersionDiscoveryData(ctx, &discovery.PutExtensionVersionDiscoveryDataRequest{
					UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
						ExtensionID: extID,
						Version:     "testVersionID",
					},
				})
				require.NoError(t, err)

				resp, err := m.AddExtensionToDeveloperCategory(ctx, &discovery.AddExtensionToCategoryRequest{
					ExtensionID: discoData.GetExtensionID(),
					CategoryID:  cat.GetID(),
					Language:    "de",
					VersionID:   twirputils.WrapString(discoData.GetVersion()),
				})
				require.NoError(t, err)
				assert.NotNil(t, resp)

				getResp, err := m.GetExtensionsForCategory(ctx, &discovery.GetCategoryExtensionsRequest{
					CategoryID: cat.GetID(),
				})

				assert.Nil(t, err)
				require.NotNil(t, getResp)
				require.Equal(t, 1, len(getResp.GetIDs()))
				assert.Equal(t, extID, getResp.GetIDs()[0])
			})
		})

		t.Run("for pseudo category", func(t *testing.T) {
			t.Run("~whitelisted calls out to EMS search~ not impl", func(t *testing.T) {
				// TODO?
			})

			t.Run("~new-releases calls out to EMS search~ not impl", func(t *testing.T) {
				// TODO?
			})

			t.Run("~other, unexpected~ not impl", func(t *testing.T) {
				// TODO?
			})
		})
	})

	t.Run("GetDiscoveryData", func(t *testing.T) {
		t.Run("data exists", func(t *testing.T) {
			ctx, m := setup(&auth.FakeCredentials{Role: auth.Categorizer})
			discoData := createDiscoveryDataWithNoErrors(t, m, validPutDiscoveryDataRequest())
			resp, err := m.GetExtensionVersionDiscoveryData(ctx, &discovery.GetExtensionVersionDiscoveryDataRequest{
				ExtensionID: discoData.GetExtensionID(),
				Version:     discoData.GetVersion(),
			})
			assert.NoError(t, err)
			assert.NotNil(t, resp)
		})

		t.Run("get store failure", func(t *testing.T) {
			store := &modelfakes.FakeStore{}
			store.GetDiscoveryDataStub = func(request *discovery.GetExtensionVersionDiscoveryDataRequest) (*model.ExtensionDiscoveryData, error) {
				return nil, errTestError
			}
			m := New(store)

			resp, err := m.GetExtensionVersionDiscoveryData(context.Background(), &discovery.GetExtensionVersionDiscoveryDataRequest{})

			assert.Nil(t, resp)
			assert.Equal(t, errTestError, err)
			assert.Equal(t, 1, store.GetDiscoveryDataCallCount())
		})

		t.Run("data does not exist", func(t *testing.T) {
			ctx, m := setup(&auth.FakeCredentials{Role: auth.Categorizer})
			resp, err := m.GetExtensionVersionDiscoveryData(ctx, &discovery.GetExtensionVersionDiscoveryDataRequest{
				ExtensionID: "id",
				Version:     "version",
			})
			assert.Error(t, err)
			assert.Nil(t, resp)
			assert.Equal(t, data.ErrNotFound, err)
		})
	})

}

func validAddCategoryRequest() *discovery.AddCategoryRequest {
	return &discovery.AddCategoryRequest{
		Category: &discovery.CategoryUpsert{
			Order:    twirputils.WrapFloat(2),
			Readonly: twirputils.WrapBool(false),
			SortKey:  twirputils.WrapStringPointer(&model.CategorySortkeyManual),
			Type:     twirputils.WrapStringPointer(&model.CategoryTypeDeveloper),
			Visible:  twirputils.WrapBool(true),
		},
	}
}

func validUpdateCategoryRequest(id string) *discovery.UpdateCategoryRequest {
	return &discovery.UpdateCategoryRequest{
		CategoryID: id,
		Category: &discovery.CategoryUpsert{
			Order:    twirputils.WrapFloat(2),
			Readonly: twirputils.WrapBool(false),
			SortKey:  twirputils.WrapStringPointer(&model.CategorySortkeyManual),
			Type:     twirputils.WrapStringPointer(&model.CategoryTypeDeveloper),
			Visible:  twirputils.WrapBool(true),
		},
	}
}

func validEditCategoryTranslationRequest(categoryID, language string) *discovery.EditCategoryTranslationRequest {
	return &discovery.EditCategoryTranslationRequest{
		Name:        "new_name",
		Description: "new_description",
		CategoryID:  categoryID,
		Language:    language,
	}
}

func validPutDiscoveryDataRequest() *discovery.PutExtensionVersionDiscoveryDataRequest {
	return &discovery.PutExtensionVersionDiscoveryDataRequest{
		UpdateData: &discovery.ExtensionVersionDiscoveryDataUpdate{
			ExtensionID: "testExtensionID",
			Version:     "testVersionID",
		},
	}
}

func setTempCreds(m interfaces.DiscoveryManager, creds auth.Credentials) context.Context {
	return auth.Store(context.Background(), creds)
}

func createCategoryWithNoErrors(t *testing.T, m interfaces.DiscoveryManager, req *discovery.AddCategoryRequest) *discovery.CategoryDocument {
	creds := &auth.FakeCredentials{Role: auth.Categorizer}
	ctx := setTempCreds(m, creds)

	cat, err := m.CreateCategory(ctx, req)
	require.NoError(t, err)
	require.NotNil(t, cat)
	return cat
}

func createDiscoveryDataWithNoErrors(t *testing.T, m interfaces.DiscoveryManager, req *discovery.PutExtensionVersionDiscoveryDataRequest) *discovery.ExtensionVersionDiscoveryDocument {
	creds := &auth.FakeCredentials{Role: auth.Categorizer}
	ctx := setTempCreds(m, creds)

	discoData, err := m.PutExtensionVersionDiscoveryData(ctx, req)
	require.NoError(t, err)
	require.NotNil(t, discoData)
	return discoData
}

func createAndSoftDeleteCategoryWithNoErrors(t *testing.T, m interfaces.DiscoveryManager, req *discovery.AddCategoryRequest) *discovery.CategoryDocument {
	creds := &auth.FakeCredentials{Role: auth.Categorizer}

	req.Category.Readonly = twirputils.WrapBool(true)
	req.Category.Visible = twirputils.WrapBool(false)
	ctx := setTempCreds(m, creds)
	cat, err := m.CreateCategory(ctx, req)
	require.NoError(t, err)
	require.NotNil(t, cat)

	ctx = setTempCreds(m, creds)
	delcat, err := m.DeleteCategory(ctx, &discovery.DeleteCategoryRequest{
		CategoryID: cat.ID,
		SoftDelete: true,
	})
	require.NoError(t, err)
	require.NotNil(t, delcat)

	return delcat.GetCategory()
}
