variable "discovery_images_cdn_hostname" {
  default = "extensions-discovery-images.twitch.tv"
}

data "aws_acm_certificate" "discovery_images_cdn_ssl_cert" {
  domain   = var.discovery_images_cdn_hostname
  statuses = ["ISSUED"]
  provider = aws.us_east_1
}

resource "aws_cloudfront_distribution" "discovery_images_cdn" {
  origin {
    domain_name = "${aws_s3_bucket.discovery_images.bucket}.s3.amazonaws.com"
    origin_id   = aws_s3_bucket.discovery_images.bucket
  }

  enabled = true
  aliases = [var.discovery_images_cdn_hostname]

  default_cache_behavior {
    allowed_methods = ["HEAD", "GET", "OPTIONS"]
    cached_methods  = ["HEAD", "GET", "OPTIONS"]

    //
    // this needs to match the origin_id created above in the origin {} block
    //
    target_origin_id = aws_s3_bucket.discovery_images.bucket

    forwarded_values {
      query_string = false

      cookies {
        forward = "none"
      }

      headers = [
        "Origin",
      ]
    }

    compress = true

    viewer_protocol_policy = "https-only"
    min_ttl                = 660 # 11 minutes
    default_ttl            = 660 # is somewhere between
    max_ttl                = 660 # 5 minutes and an hour
  }

  price_class = "PriceClass_200"

  restrictions {
    geo_restriction {
      restriction_type = "none"
    }
  }

  viewer_certificate {
    acm_certificate_arn      = data.aws_acm_certificate.discovery_images_cdn_ssl_cert.arn
    minimum_protocol_version = "TLSv1"
    ssl_support_method       = "sni-only"
  }

  tags = {
    "twitch:app:name"           = var.service
    "twitch:app:env"            = var.environment
    "twitch:app:component_name" = var.component
    "twitch:cape:owner"         = var.owner
  }
}

output "cdn_hostname" {
  value = aws_cloudfront_distribution.discovery_images_cdn.domain_name
}

