package rule

import (
	"fmt"
	"math"
	"strings"
	"time"
)

// C7Value is a union type for a given value node
type C7Value struct {
	StrValue      *string
	IntValue      *int64
	BoolValue     *bool
	DurationValue *time.Duration
}

// StrType returns the nice type of the value
func (c C7Value) StrType() string {
	if c.IntValue != nil {
		return "int64"
	}
	if c.StrValue != nil {
		return "string"
	}
	if c.BoolValue != nil {
		return "bool"
	}
	return "none"
}

// Rule represents one rule in the config system
type Rule struct {
	Selectors []string
	Namespace string
	Keys      []string // The keys, keeping as a array for future expansion
	Operator  string

	Value C7Value
}

// String converts this to a string, outputted in the extended format for writing to a file
func (r *Rule) String() string {
	var strValue string
	if r.Value.IntValue != nil {
		strValue = fmt.Sprintf("%d", *r.Value.IntValue)
	} else if r.Value.StrValue != nil {
		strValue = fmt.Sprintf("\"%s\"", *r.Value.StrValue)
	} else if r.Value.BoolValue != nil {
		if *r.Value.BoolValue {
			strValue = "true"
		} else {
			strValue = "false"
		}
	} else if r.Value.DurationValue != nil {
		// We currently only allow in whole ms, s, m, or h, and not mixed units, so
		// convert back to the Lowest common denomiator
		d := *r.Value.DurationValue
		if (d.Nanoseconds()/1000000)%1000 != 0 {
			strValue = fmt.Sprintf("%dms", d.Nanoseconds()/1000000)
		} else if d.Hours() == math.Trunc(d.Hours()) {
			strValue = fmt.Sprintf("%dh", int64(d.Hours()))
		} else if d.Minutes() == math.Trunc(d.Minutes()) {
			strValue = fmt.Sprintf("%dm", int64(d.Minutes()))
		} else {
			strValue = fmt.Sprintf("%ds", int64(d.Seconds()))
		}
	}
	return fmt.Sprintf("%s = %s;", r.PrefixString(), strValue)
}

// StrType convience method to get the type from the value object
func (r *Rule) StrType() string {
	return r.Value.StrType()
}

// PrefixString is lhs of a rule, eg Selectors:Namespace:Keys
func (r *Rule) PrefixString() string {
	selectors := strings.Join(r.Selectors, ".")
	keys := strings.Join(r.Keys, ".")
	return fmt.Sprintf("%s:%s:%s", selectors, r.Namespace, keys)
}

// Matches returns true if this rule matches with the namespace and keys specified
func (r *Rule) Matches(namespace string, keys ...string) bool {
	if r.Namespace != namespace {
		return false
	}

	if !strAryEql(r.Keys, keys) {
		return false
	}
	return true
}

// Rules is a collection of Rules
type Rules []*Rule

// String returns a file compatible representation of rules
func (r Rules) String() string {
	strOut := ""
	for _, r := range r {
		strOut += r.String() + "\n"
	}
	return strOut
}

func strAryEql(s1, s2 []string) bool {
	if len(s1) != len(s2) {
		return false
	}
	for i, v := range s1 {
		if v != s2[i] {
			return false
		}
	}
	return true
}

// EqualsIgnoringValue returns true if the lhs on a rule is equal to other lhs, useful for aggregation
func (r *Rule) EqualsIgnoringValue(other *Rule) bool {
	if r.Namespace != other.Namespace {
		return false
	}
	if !strAryEql(r.Selectors, other.Selectors) {
		return false
	}
	if !strAryEql(r.Keys, other.Keys) {
		return false
	}
	return true
}
