package logmiddleware

import (
	logging "code.justin.tv/amzn/TwitchLogging"
	"code.justin.tv/video/metricmiddleware-beta/operation"
	"context"
	"time"
)

// OperationMonitor monitors operation spans and logs them as necessary
type OperationMonitor struct {
	// Logger used by MonitorOp. It is expected that this logger already contain keyvals about the service (such as
	// ServiceTuples).
	Logger logging.Logger
	// TODO: Consider adding multiple Loggers (so we can have a Debug logger, for example)
}

const (
	// An RFC3339 format with Millisecond precision
	RFC3339Millis = "2006-01-02T15:04:05.000Z07:00"
)

const (
	groupKey  = "OpGroup"
	methodKey = "OpMethod"
	kindKey   = "OpKind"
	startKey  = "OpStartTime"
	endKey    = "OpEndTime"
	statusKey = "OpStatusCode"
	pointKey  = "OpPoint"
	end       = "End"
)

// MonitorOp monitors operation spans and processes their reports when complete, logging the results
func (m *OperationMonitor) MonitorOp(ctx context.Context, name operation.Name) (context.Context, *operation.MonitorPoints) {
	monitorPoints := &operation.MonitorPoints{}

	monitorPoints.End = func(report *operation.Report) {
		m.writeMonitorOpEndLog(name.Kind.String(), name, report)
	}

	return ctx, monitorPoints
}

// writeMonitorOpEndLog constructs a log with set keyvals. The log msg is "MonitorOp" to reflect where this log is
// coming from. The key/val pairs express the kind of operation, the point when the log is written (the end of the
// operation), and operation.Name (Group, Method) and report (StartTime, EndTime, Status.Code) information.
func (m *OperationMonitor) writeMonitorOpEndLog(kind string, name operation.Name, report *operation.Report) {
	if m.Logger != nil {
		m.Logger.Log("MonitorOp",
			kindKey, kind,
			pointKey, end,
			groupKey, name.Group,
			methodKey, name.Method,
			startKey, formatTime(report.StartTime),
			endKey, formatTime(report.EndTime),
			statusKey, statusNameFromGRPCCode(report.Status.Code))
	}
}

// formatTime takes a time and returns the log-friendly RFC3339-formatted timestamp, showing the given time with
// millisecond accuracy.
func formatTime(timestamp time.Time) string {
	return timestamp.Format(RFC3339Millis)
}

var statusNameByGRPCCode = map[int32]string{
	0:  "ok",
	1:  "canceled",
	2:  "unknown",
	3:  "invalid_argument",
	4:  "deadline_exceeded",
	5:  "not_found",
	6:  "already_exists",
	7:  "permission_denied",
	8:  "resource_exhausted",
	9:  "failed_precondition",
	10: "aborted",
	11: "out_of_range",
	12: "unimplemented",
	13: "internal",
	14: "unavailable",
	15: "data_loss",
	16: "unauthenticated",
}

func statusNameFromGRPCCode(code int32) string {
	name, ok := statusNameByGRPCCode[code]
	if !ok {
		name = "unknown"
	}
	return name
}
