package auth

import (
	"context"
	"net/http"

	"github.com/jixwanwang/apiutils"
)

var (
	// this value is never used, the address is important
	credentialsMarker = 42
	// credentialsKey is a unique address within this process and private to this
	// package so that it can never be accidentally stepped on by other values.
	credentialsKey = &credentialsMarker
)

// Load retrieves Credentials from a context defaulting to NoPermissions()
func Load(ctx context.Context) Credentials {
	if cast, ok := ctx.Value(credentialsKey).(Credentials); ok {
		return cast
	}
	return NoPermissions()
}

// Store places Credentials into a context
func Store(ctx context.Context, creds Credentials) context.Context {
	return context.WithValue(ctx, credentialsKey, creds)
}

// StoreHandler builds a goji middleware that automatically places a
// Credentials into the context of each request or early outs with an
// unauthorized error. Note that EMS supplies its own version of this
// function that already injects a superset of Credentials, making
// this middleware redundant in that program.
func StoreHandler(handler Handler) func(http.Handler) http.Handler {
	return func(inner http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			creds, err := handler.GetCredentials(r)
			if err != nil {
				apiutils.ServeError(w, apiutils.NewErrorResponse(http.StatusUnauthorized, err.Error()))
			} else {
				ctx := Store(r.Context(), creds)
				inner.ServeHTTP(w, r.WithContext(ctx))
			}
		})
	}
}
