package protocol

// Action is a verb that our whitelists track
type Action string

const (
	// CreateExtensions allows creation of new extensions
	CreateExtensions = Action("create_extensions")
	// CurateAllCategories allows users to update category and game membership
	CurateAllCategories = Action("curate_all_categories")
	// DeleteAllJwtSecrets allows deletion of any extension's shared secret
	DeleteAllJwtSecrets = Action("delete_all_jwt_secrets")
	// DisplayOnIOS allows an extension to show its mobile version on iOS
	DisplayOnIOS = Action("display_on_ios")
	// EditAllCategories allows modification of categories or their version association
	EditAllCategories = Action("edit_all_categories")
	// EditAllWhitelists allows modification of any whitelist membership
	EditAllWhitelists = Action("edit_all_whitelists")
	// HardDeleteExtensions allows hard deletes of extensions
	HardDeleteExtensions = Action("hard_delete_extensions")
	// InstallAllExtensions allows installation of any visible extension
	InstallAllExtensions = Action("install_all_extensions")
	// LinkExternalContent allows an extension to outbound link
	LinkExternalContent = Action("link_external_content")
	// ModerateExtensions allows rejection of live extensions
	ModerateExtensions = Action("moderate_extensions")
	// MonetizeExtensions allows developers to monetize their extension
	MonetizeExtensions = Action("monetize_extensions")
	// ReviewExtensions allows approval or denial of extension versions in review
	ReviewExtensions = Action("review_extensions")
	// ReviveExtensions allows any deleted extension to be restored
	ReviveExtensions = Action("revive_extensions")
	// ViewAllExtensions allows the ability to see any active (not rejected) extension
	ViewAllExtensions = Action("view_all_extensions")
	// ViewAllJwtSecrets allows the ability to see any extension's shared secret
	ViewAllJwtSecrets = Action("view_all_jwt_secrets")
	// ViewAllWhitelists allows viewing any whitelist membership
	ViewAllWhitelists = Action("view_all_whitelists")

	// EditWhitelistPrefix is used as a prefix for editor rights for specific whitelists
	EditWhitelistPrefix = "edit_whitelist"
	// ViewWhitelistPrefix is used as a prefix for viewer rights for specific whitelists
	ViewWhitelistPrefix = "view_whitelist"
)

// CurrentActionMap returns the cache setting for known actions; although clients
// may use this function to see the status of the map at their compile time, they
// can also query the server for an up-to-date version; this is the preferred
// method.
func CurrentActionMap() ActionMap {
	cache := ActionSettings{Cache: true}
	noCache := ActionSettings{Cache: false}
	base := ActionMap{
		CreateExtensions:     noCache,
		CurateAllCategories:  cache,
		DisplayOnIOS:         cache,
		EditAllWhitelists:    cache,
		EditAllCategories:    cache,
		HardDeleteExtensions: noCache,
		InstallAllExtensions: cache,
		LinkExternalContent:  cache,
		ModerateExtensions:   cache,
		MonetizeExtensions:   noCache,
		ReviewExtensions:     cache,
		ReviveExtensions:     cache,
		ViewAllExtensions:    cache,
		ViewAllWhitelists:    cache,
	}
	total := make(ActionMap)
	for action, value := range base {
		total[action] = value
		total[ViewWhitelist(action)] = cache
		if action != EditAllWhitelists {
			// there is no point in having a distinct permission for editing the edit
			// all list, so don't offer one.
			total[EditWhitelist(action)] = cache
		}
	}
	return total
}

// EditWhitelist is a wrapper that builds an action for the edit permissions of
// a particular whitelist. It is not available recursively.
func EditWhitelist(action Action) Action {
	return Action(EditWhitelistPrefix + ":" + string(action))
}

// ViewWhitelist is a wrapper that builds an action for the view permissions of
// a particular whitelist. It is not available recursively
func ViewWhitelist(action Action) Action {
	return Action(ViewWhitelistPrefix + ":" + string(action))
}

// In allows searching for an Action in an array
func (a Action) In(arr []Action) bool {
	for _, v := range arr {
		if v == a {
			return true
		}
	}
	return false
}
