// Package ephemeralcert can be used to generate ephemeral self-signed certificates for use within Go's TLS library. It
// has the notable benefit that you don't need to pre-generate a certificate (e.g. with OpenSSL) and read it in from the
// filesystem; instead, you can generate an ephemeral cert in-memory at runtime.
//
// The intended use case is for encryption between a TLS-terminating load balancer (either an ALB, NLB, or classic ELB)
// and your container- or instance-based service. It's important to note that in most traditional networking
// environments, use of self-signed certificates in production is insecure and unwise. However, an AWS VPC provides
// strong network integrity guarantees, so certificate validation is less useful and the benefit of encrypting traffic
// and eliminating the risk of certificate management makes it a worthwhile tradeoff.
package ephemeralcert // import "code.justin.tv/security/ephemeralcert"

import (
	"crypto/rand"
	"crypto/rsa"
	"crypto/tls"
	"crypto/x509"
	"crypto/x509/pkix"
	"math/big"
	"time"
)

const commonName = "localhost"
const rsaKeySize = 2048

// GenerateEphemeralCert generates a new ephemeral self-signed certificate suitable for a TLS configuration.
func GenerateEphemeralCert() (tls.Certificate, error) {
	priv, err := rsa.GenerateKey(rand.Reader, rsaKeySize)
	if err != nil {
		return tls.Certificate{}, err
	}

	notBefore := time.Now()
	notAfter := notBefore.Add(time.Hour * 24 * 365 * 2) // 2 years

	serialNumberLimit := new(big.Int).Lsh(big.NewInt(1), 128)
	serialNumber, err := rand.Int(rand.Reader, serialNumberLimit)
	if err != nil {
		return tls.Certificate{}, err
	}

	entity := pkix.Name{
		CommonName: commonName,
	}

	template := x509.Certificate{
		SerialNumber:          serialNumber,
		Issuer:                entity,
		Subject:               entity,
		NotBefore:             notBefore,
		NotAfter:              notAfter,
		KeyUsage:              x509.KeyUsageKeyEncipherment | x509.KeyUsageDigitalSignature | x509.KeyUsageCertSign,
		BasicConstraintsValid: true,
		IsCA:                  true,
		DNSNames:              []string{commonName},
	}

	rawCert, err := x509.CreateCertificate(rand.Reader, &template, &template, &priv.PublicKey, priv)
	if err != nil {
		return tls.Certificate{}, err
	}

	return tls.Certificate{
		Certificate: [][]byte{rawCert},
		PrivateKey:  priv,
	}, nil
}

// GenerateTLSConfig constructs a new TLS configuration that contains a newly generated ephemeral self-signed
// certificate. It is suitable for use in a TLS listener.
func GenerateTLSConfig() (*tls.Config, error) {
	cert, err := GenerateEphemeralCert()
	if err != nil {
		return nil, err
	}

	return &tls.Config{Certificates: []tls.Certificate{cert}}, nil
}
