import json
import urllib.parse
import boto3

DYNAMO_PARTITION_KEY = ''
DYNAMO_SORT_KEY = ''
DYNAMO_TABLE_NAME = ''

s3 = boto3.client('s3')
dynamo = boto3.client('dynamodb')

def get_s3_object(bucket, key):
  try:
    response = s3.get_object(Bucket=bucket, Key=key)
    return json.loads(response['Body'].read().decode('utf-8'))
  except Exception as e:
    print(e)
    print('Error getting object {} from bucket {}. Make sure they exist and your bucket is in the same region as this function.'.format(key, bucket))
    raise e
  
def get_dynamo_key(resource):
  dynamo_key = {}
  dynamo_key[DYNAMO_PARTITION_KEY] = resource['dynamodb'][DYNAMO_PARTITION_KEY]
  dynamo_key[DYNAMO_SORT_KEY] = resource['dynamodb'][DYNAMO_SORT_KEY]
  return dynamo_key

def update_record(resource):
  insert_item = resource['dynamodb']['NewImage']
  return dynamo.put_item(TableName=DYNAMO_TABLE_NAME, Item=insert_item)

def delete_record(resource):
  dynamo_key = get_dynamo_key(resource)
  return dynamo.delete_item(TableName=DYNAMO_TABLE_NAME, Key=dynamo_key)

def update_dynamo_table(resource):
  operation = resource['eventName']
  # We can use put_item for both insert and modify as we have the whole image
  if operation == 'INSERT' or operation == 'MODIFY':
    return update_record(resource)
  elif operation == 'REMOVE':
    return delete_record(resource)
  else:
    print('Error handling operation. Unsupported operation: {}'.format(operation))
    return 'Unsupported Operation'


def lambda_handler(event, context):
    # Get the object from the event and show its content type
    bucket = event['Records'][0]['s3']['bucket']['name']
    key = urllib.parse.unquote_plus(event['Records'][0]['s3']['object']['key'], encoding='utf-8')
    
    resource = get_s3_object(bucket, key)
    update_response = update_dynamo_table(resource)

    return 'success'

