#!/usr/bin/env groovy

import groovy.json.JsonOutput

INTEGRATION_STAGING = "http://dev-graphql.twitch.a2z.com"
AWS_REGION = "us-west-2"
DEV_CREDENTIALS_ID = "twitch-gds-dev"

def FIRST_ALERT = 6
def REPEAT_ALERT = 144

pipeline {
    agent any
    triggers {
        cron('H/10 * * * *')
    }

    options {
        ansiColor('xterm')
    }

    stages {
        stage("Sanity Check") {
            steps {
                validateDeclarativePipeline 'Jenkinsfile'
                sh 'printenv'
            }
        }

        stage("Run Eastwatch Tests") {
            steps {
                checkout([$class: 'GitSCM',
                branches: [[name: '*/master']],
                doGenerateSubmoduleConfigurations: false,
                extensions: [],
                submoduleCfg: [],
                userRemoteConfigs: [[credentialsId: 'git-aws-read-key', url: 'git@git-aws.internal.justin.tv:extensions/eastwatch.git']]])
                withAWS(credentials: DEV_CREDENTIALS_ID, region: AWS_REGION) {
                    sh 'bin/run_tests_cron'
                }

                checkout([$class: 'GitSCM',
                branches: [[name: '*/master']],
                doGenerateSubmoduleConfigurations: false,
                extensions: [],
                submoduleCfg: [],
                userRemoteConfigs: [[credentialsId: 'git-aws-read-key', url: 'git@git-aws.internal.justin.tv:gds/gds.git']]])
            }
        }
    }

    post {
        failure {
            script {
                int failedBuilds = 1
                def build = currentBuild.getPreviousBuild()


                while(build != null && build.result == 'FAILURE') {
                    failedBuilds++
                    build = build.getPreviousBuild()
                }

                if( failedBuilds == FIRST_ALERT || ((failedBuilds % REPEAT_ALERT) == 0)) {

                    def changeSets = build.getNextBuild().changeSets
                    def blameList = [];

                    for(int i = 0; i < changeSets.size(); i++) {
                        for(int j = 0; j < changeSets[i].items.length; j++) {
                            def entry = changeSets[i].items[j]
                            blameList << entry.msg
                        }
                    }

                    def title = "Eastwatch has failed ${failedBuilds} times"
                    def msg = "${env.BUILD_URL}testReport/\n";

                    if ( blameList.size() > 0 ) {
                        msg += "Commits: \n"
                        for( int i = 0; i < blameList.size(); i ++ ) {
                            msg += blameList[i] + "\n"
                        }
                    } else {
                        msg += "No changes detected"
                    }

                    sendNotification("#dx-ext-disco-oncall", ":ohno3:", title, msg)
                    sendNotification("#ext-anchors-oncall", ":ohno3:", title, msg)
                }
            }
        }
        success {
            script {
                def build = currentBuild.getPreviousBuild()
                def i = 0

                while( i < FIRST_ALERT && build != null && build.result == 'FAILURE' ) {
                    i++
                    build = build.getPreviousBuild()
                }

                if( i >= FIRST_ALERT ) {
                    def title = "Eastwatch recovered!"
                    def msg = ""

                    sendNotification("#dx-ext-disco-oncall", ":ohno6:", title, msg)
                    sendNotification("#ext-anchors-oncall", ":ohno6:", title, msg)
                }
            }
        }
        always {
            junit '**/testoutput/report.xml'
        }
    }
}

void sendNotification(String channel, String emoji, String title, String msg) {
    withCredentials([string(credentialsId: 'eastwatch-slack-anchors-url', variable: 'URL')]) {
        def json = JsonOutput.toJson([
            text: '',
            channel: "${channel}",
            username: "Eastwatch",
            icon_emoji: "${emoji}",
            attachments: [[
                fields: [[
                    title: "${title}",
                    value: "${msg}",
                ]]
            ]]
        ])

        writeFile(file: 'payload.json', text: json)
        sh('curl --fail -X POST --data-urlencode payload@payload.json $URL')
    }
}
