package cartman_test

import (
	"context"
	"fmt"
	"net/http"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/users"

	"code.justin.tv/extensions/eastwatch/internal/api/cartman"
	"code.justin.tv/extensions/eastwatch/internal/api/cartman/cartmanfakes"
	"code.justin.tv/extensions/eastwatch/internal/config"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/foundation/twitchclient"
	cartmanclient "code.justin.tv/web/cartman/client"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestGetToken(t *testing.T) {

	fakeOAuth, err := token.NewOAuth("fake-oauth")
	require.NoError(t, err)
	fakeChannelID := "fake-channel_id"
	fakeExtID := "fake-extension_id"
	fakeExtVer := "fake-extension_version"
	fakeJWT := "fake-cartman_jwt"

	cartmanClient := new(cartmanfakes.FakeCartmanClient)
	cartmanClient.GetAuthorizationTokenReturns(&cartmanclient.TokenResponse{Token: fakeJWT}, nil)
	client := cartman.NewClient(cartmanClient)
	// cartman.SetClientForTesting(fakeCartmanURL, cartmanClient)

	caps := []string{
		cartman.CapExtensionsChannelActivations,
		cartman.CapExtensionsChannelInstalls,
	}
	jwt, err := client.GetToken(
		fakeOAuth,
		caps,
		cartman.ForChannel(fakeChannelID),
		cartman.ForExtension(fakeExtID, fakeExtVer),
	)

	require.Equal(t, cartmanClient.GetAuthorizationTokenCallCount(), 1, "expecting GetAuthorizationToken to be called once")
	ctx, params, body := cartmanClient.GetAuthorizationTokenArgsForCall(0)

	t.Run("it should not error", func(t *testing.T) {
		assert.NoError(t, err)
	})

	t.Run("it should return the jwt from the cartman response", func(t *testing.T) {
		assert.Equal(t, fakeJWT, jwt.String())
	})

	t.Run("it should call the cartman client with an empty context", func(t *testing.T) {
		assert.Equal(t, context.Background(), ctx)
	})

	t.Run("it should have an empty body", func(t *testing.T) {
		assert.Nil(t, body)
	})

	t.Run("it should set the correct capabilities", func(t *testing.T) {
		assert.ElementsMatch(t, caps, params.Capabilities)
	})

	t.Run("it should set the correct parameters", func(t *testing.T) {
		assert.Equal(t, map[string]string{
			cartmanclient.KeyParam: "ecc.key",
			"channel_id":           fakeChannelID,
			"extension_id":         fakeExtID,
			"extension_version":    fakeExtVer,
		}, params.Params)
	})

	t.Run("it should set the correct origin request", func(t *testing.T) {
		o := params.Origin
		assert.Equal(t, http.MethodGet, o.Method)
		assert.Equal(t, "/", o.URL.String())
		assert.Nil(t, o.Body)
	})
}

func TestRealGetTokenCall(t *testing.T) {
	t.Skip("skipping because it makes a real cartman call, unskip to see example jwt returned from cartman")

	cfg := config.Staging
	oauth, err := users.User3.OAuthToken(cfg, config.SiteDevsite)
	require.NoError(t, err)

	cartmanClient, err := cartmanclient.NewClient(twitchclient.ClientConf{Host: cfg.CartmanURL})
	require.NoError(t, err)
	client := cartman.NewClient(cartmanClient)

	jwt, err := client.GetToken(
		oauth,
		[]string{
			cartman.CapExtensionsChannelActivations,
			cartman.CapExtensionsChannelInstalls,
		},
		cartman.ForChannel(users.User3.ID()),

		//eastwatch-vmxj-devsite
		cartman.ForExtension("tq0mtb21k458cqwukpnpd429046l81", "0.0.1"),
	)
	require.NoError(t, err)
	require.NotEmpty(t, jwt)
	fmt.Println(jwt)
}
