package ems

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"path"

	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/gds/gds/extensions/ems/documents"
)

// ActivateExtensionRequest define the required details needed to install an extension via EMS
type ActivateExtensionRequest struct {
	ChannelID        string
	ExtensionID      string
	ExtensionVersion string
	Slot             Slot
}

// Slot specifies an extension slot on a channel
type Slot string

const (
	// SlotExtensionPanel1 is the left extension panel
	SlotExtensionPanel1 Slot = "extension-panel-1"

	// SlotExtensionPanel2 is the middle extension panel
	SlotExtensionPanel2 Slot = "extension-panel-2"

	// SlotExtensionPanel3 is the right extension panel
	SlotExtensionPanel3 Slot = "extension-panel-3"
)

func (s Slot) String() string {
	return string(s)
}

func (c *client) ActivateExtension(jwt token.JWT, r ActivateExtensionRequest) error {
	// PUT /channels/:cid/extensions/activations
	path, err := url.Parse(path.Join("channels", r.ChannelID, "extensions/activations"))
	if err != nil {
		return err
	}

	doc := documents.ActivationConfigurationParams{
		Activations: []documents.ActivationConfigurationParam{
			{
				ID:      r.ExtensionID,
				Version: r.ExtensionVersion,
				Anchor: documents.AnyAnchorParam{
					Value: &documents.PanelAnchorActivationParam{
						SlotName: r.Slot.String(),
					},
				},
			},
		},
	}
	body, err := json.Marshal(doc)
	if err != nil {
		return err
	}

	req, err := c.buildRequest(context.Background(), "ActivateExtension", http.MethodPut, c.baseURL.ResolveReference(path).String(), bytes.NewBuffer(body))
	if err != nil {
		return err
	}
	req.Header.Set(twitchclient.TwitchAuthorizationHeader, jwt.String())

	resp, err := c.http.Do(req)
	if err != nil {
		return err
	}

	if resp.StatusCode != http.StatusOK {
		return fmt.Errorf("activate extension unsuccessful [%s]", resp.Status)
	}

	return nil
}
