package ems_test

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/ems"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func activateExtensionHandler(returnStatusCode int, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(returnStatusCode)
	})
	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestActivateExtension_Successful(t *testing.T) {
	jwt, err := token.NewJWT("fake-jwt")
	require.NoError(t, err)
	r := ems.ActivateExtensionRequest{
		ChannelID:        "63728",
		ExtensionID:      "83712",
		ExtensionVersion: "3.4.5",
		Slot:             ems.SlotExtensionPanel2,
	}

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(activateExtensionHandler(http.StatusOK, &captured))
	defer server.Close()

	client, err := ems.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	err = client.ActivateExtension(jwt, r)

	t.Run("it should not error", func(t *testing.T) {
		require.NoError(t, err)
	})

	t.Run("it should setup the method and url correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodPut, req.Method)
		assert.Equal(t, "/channels/63728/extensions/activations", req.URL.String())
	})

	t.Run("it should set the correct header values", func(t *testing.T) {
		assert.Equal(t, jwt.String(), captured.Request.Header.Get("Twitch-Authorization"))
	})

	t.Run("it should setup the body created correctly", func(t *testing.T) {
		var actual, expected map[string]interface{}
		require.NoError(t, testutil.UnmarshalTestData("activate_extension_request.json", &expected))
		require.NoError(t, json.Unmarshal(captured.Body, &actual))
		assert.Equal(t, expected, actual)
	})
}

func TestActivateExtension_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(activateExtensionHandler(http.StatusNotFound, nil))
	defer server.Close()

	client, err := ems.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	err = client.ActivateExtension("", ems.ActivateExtensionRequest{})

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "activate extension unsuccessful [404 Not Found]", err.Error())
	})
}
