package ems_test

import (
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/assert"

	"github.com/stretchr/testify/require"

	"code.justin.tv/extensions/eastwatch/internal/api/ems"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
)

func adminDeactivateExtensionHandler(returnStatusCode int, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(returnStatusCode)
	})
	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestAdminDeactivateExtension_Successful(t *testing.T) {
	fakeChannelID := "12347"
	fakeAnchorType := ems.Panel
	fakeSlot := ems.SlotExtensionPanel1

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(adminDeactivateExtensionHandler(http.StatusNoContent, &captured))
	defer server.Close()

	client, err := ems.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	err = client.AdminDeactivateExtension(fakeChannelID, fakeAnchorType, fakeSlot)

	t.Run("it should not error", func(t *testing.T) {
		require.NoError(t, err)
	})

	t.Run("it should set up the method and url correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodDelete, req.Method)
		assert.Equal(t, "/admin/channels/12347/panel/extension-panel-1", req.URL.String())
	})

	t.Run("it should have an empty body", func(t *testing.T) {
		assert.Empty(t, captured.Body)
	})

}

func TestAdminDeactivateExtension_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(adminDeactivateExtensionHandler(http.StatusUnprocessableEntity, nil))
	defer server.Close()

	client, err := ems.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	err = client.AdminDeactivateExtension("", "", "")

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "admin deactivate extension unsuccessful [422 Unprocessable Entity]", err.Error())
	})
}
