package ems_test

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/ems"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"code.justin.tv/gds/gds/extensions/ems/documents"
	"code.justin.tv/gds/gds/extensions/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func adminDumpExtensionHandler(returnStatusCode int, doc documents.AdminExtensionsDocument, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if returnStatusCode != http.StatusOK {
			w.WriteHeader(returnStatusCode)
		}
		b, err := json.Marshal(doc)
		if err != nil {
			panic(err)
		}
		w.Write(b)
	})
	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestAdminDumpExtensions_Successful(t *testing.T) {
	reqDoc := documents.GetExtensionsParams{
		Searches: []documents.SearchTerm{
			{
				Field: "id",
				Term:  "abcde",
			},
		},
	}

	respDoc := documents.AdminExtensionsDocument{
		Count: 1,
		Extensions: []*models.Extension{
			{
				ID:      "abcde",
				Version: "0.1.2",
			},
		},
	}

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(adminDumpExtensionHandler(http.StatusOK, respDoc, &captured))
	defer server.Close()

	client, err := ems.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	resp, err := client.AdminDumpExtensions(reqDoc)

	t.Run("it should not error", func(t *testing.T) {
		require.NoError(t, err)
	})

	t.Run("it should setup the method and URL correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodPost, req.Method)
		assert.Equal(t, "/admni/extensions", req.URL.String())
	})

	t.Run("it should have the correct body", func(t *testing.T) {
		var actual documents.GetExtensionsParams
		require.NoError(t, json.Unmarshal(captured.Body, &actual))
		assert.Equal(t, reqDoc, actual)
	})

	t.Run("it should return the right result", func(t *testing.T) {
		assert.Equal(t, resp, respDoc)
	})

}

func TestAdminDumpExtensions_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(adminDumpExtensionHandler(http.StatusNotFound, documents.AdminExtensionsDocument{}, nil))
	defer server.Close()

	client, err := ems.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	_, err = client.AdminDumpExtensions(documents.GetExtensionsParams{})

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "admin dump extensions unsuccessful [404 Not Found]", err.Error())
	})
}
