package ems

import (
	"context"
	"io"
	"net/http"
	"net/url"

	"code.justin.tv/extensions/eastwatch/internal/metrics"

	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/gds/gds/extensions/ems/documents"
)

//go:generate counterfeiter . Client

// Client exposes behavior available in EMS
type Client interface {
	// InstallExtension will install an extension onto a channel
	InstallExtension(token.JWT, InstallExtensionRequest) error

	// ActivateExtension will activate an extension installed on a channel
	ActivateExtension(token.JWT, ActivateExtensionRequest) error

	// UninstallExtension will uninstall an extension from a channel
	UninstallExtension(token.JWT, UninstallExtensionRequest) error

	// HardDeleteExtension will hard delete an extension
	HardDeleteExtension(jwt token.JWT, extensionID string) error

	// AdminDumpExtensions will dump all information about all extensions that meet criteria
	AdminDumpExtensions(doc documents.GetExtensionsParams) (documents.AdminExtensionsDocument, error)

	// AdminDeactivateExtension will deactivate an extension from the specified user's channel
	AdminDeactivateExtension(channelID string, anchorType AnchorType, slot Slot) error

	// GetChannelExtensionOAuth will return the OAuth token for a given Channel's Extension
	GetChannelExtensionOAuth(channelID, extensionID string) (documents.ExtensionInstallationOAuthResponse, error)
}

type client struct {
	http        *http.Client
	baseURL     *url.URL
	ServiceName string
}

// NewClient creates a client that can make requests to EMS
func NewClient(http *http.Client, baseURL string) (Client, error) {
	u, err := url.Parse(baseURL)
	if err != nil {
		return nil, err
	}
	return &client{
		http:        http,
		baseURL:     u,
		ServiceName: "EMS",
	}, nil
}

func (c *client) buildRequest(ctx context.Context, operationName string, method string, url string, body io.Reader) (*http.Request, error) {
	d := metrics.Dimensions{
		DependencyServiceName:   c.ServiceName,
		DependencyOperationName: operationName,
	}

	ctx = metrics.ContextWithDimensions(ctx, d)

	req, err := http.NewRequest(method, url, body)
	if err != nil {
		return nil, err
	}

	return req.WithContext(ctx), nil

}
