package ems_test

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/gds/gds/extensions/ems/documents"
	"github.com/stretchr/testify/assert"

	"code.justin.tv/extensions/eastwatch/internal/api/ems"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"github.com/stretchr/testify/require"
)

func getChannelExtensionOAuthHandler(returnStatusCode int, doc documents.ExtensionInstallationOAuthResponse, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if returnStatusCode != http.StatusOK {
			w.WriteHeader(returnStatusCode)
		}
		b, err := json.Marshal(doc)
		if err != nil {
			panic(err)
		}
		w.Write(b)
	})

	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestGetChannelExtensionOAuth_Successful(t *testing.T) {
	fakeChannelID := "12345"
	fakeExtensionID := "a1b2c3d4e5"

	respDoc := documents.ExtensionInstallationOAuthResponse{
		OAuthToken: "abcdefg",
	}

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(getChannelExtensionOAuthHandler(http.StatusOK, respDoc, &captured))
	defer server.Close()

	client, err := ems.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	resp, err := client.GetChannelExtensionOAuth(fakeChannelID, fakeExtensionID)

	t.Run("it should not error", func(t *testing.T) {
		require.NoError(t, err)
	})

	t.Run("it should set up the method and url correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodGet, req.Method)
		assert.Equal(t, "/channels/12345/extensions/a1b2c3d4e5/oauth", req.URL.String())
	})

	t.Run("it should return the OAuth token", func(t *testing.T) {
		assert.Equal(t, resp, respDoc)
	})

}

func TestGetChannelExtensionOAuth_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(getChannelExtensionOAuthHandler(http.StatusUnauthorized, documents.ExtensionInstallationOAuthResponse{}, nil))
	defer server.Close()

	client, err := ems.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	_, err = client.GetChannelExtensionOAuth("", "")

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "get extension installation oauth unsuccessful [401 Unauthorized]", err.Error())
	})
}
