package ems_test

import (
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/ems"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func hardDeleteExtensionHandler(returnStatusCode int, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(returnStatusCode)
	})
	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestHardDeleteExtension_Successful(t *testing.T) {
	jwt, err := token.NewJWT("fake-jwt")
	require.NoError(t, err)
	extensionID := "62537"

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(hardDeleteExtensionHandler(http.StatusNoContent, &captured))
	defer server.Close()

	client, err := ems.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	err = client.HardDeleteExtension(jwt, extensionID)

	t.Run("it should not error", func(t *testing.T) {
		require.NoError(t, err)
	})

	t.Run("it should setup the method and url correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodDelete, req.Method)
		assert.Equal(t, "/admin/extensions/62537", req.URL.String())
	})

	t.Run("it should set the correct header values", func(t *testing.T) {
		assert.Equal(t, jwt.String(), captured.Request.Header.Get("Twitch-Authorization"))
	})

	t.Run("it should have an empty body", func(t *testing.T) {
		assert.Empty(t, captured.Body)
	})
}

func TestHardDeleteExtension_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(hardDeleteExtensionHandler(http.StatusNotFound, nil))
	defer server.Close()

	client, err := ems.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	err = client.HardDeleteExtension("", "")

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "hard delete extension unsuccessful [404 Not Found]", err.Error())
	})
}
