package ems_test

import (
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/ems"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func installExtensionHandler(returnStatusCode int, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(returnStatusCode)
	})
	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestInstallExtensionV2_Successful(t *testing.T) {
	jwt, err := token.NewJWT("fake-jwt")
	require.NoError(t, err)
	r := ems.InstallExtensionRequest{
		ChannelID:        "92384",
		ExtensionID:      "83712",
		ExtensionVersion: "3.4.5",
	}

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(installExtensionHandler(http.StatusOK, &captured))
	defer server.Close()

	client, err := ems.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	err = client.InstallExtension(jwt, r)

	t.Run("it should not error", func(t *testing.T) {
		require.NoError(t, err)
	})

	t.Run("it should setup the method and url correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodPost, req.Method)
		assert.Equal(t, "/channels/92384/extensions/83712/3.4.5/v2", req.URL.String())
	})

	t.Run("it should set the correct header values", func(t *testing.T) {
		assert.Equal(t, jwt.String(), captured.Request.Header.Get("Twitch-Authorization"))
	})

	t.Run("it should have an empty body", func(t *testing.T) {
		assert.Empty(t, captured.Body)
	})
}

func TestInstallExtensionV2_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(installExtensionHandler(http.StatusForbidden, nil))
	defer server.Close()

	client, err := ems.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	err = client.InstallExtension("", ems.InstallExtensionRequest{})

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "install extension unsuccessful [403 Forbidden]", err.Error())
	})
}
