package ems_test

import (
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/ems"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func uninstallExtensionHandler(returnStatusCode int, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(returnStatusCode)
	})
	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestUninstallExtension_Successful(t *testing.T) {
	jwt := token.JWT("fake-jwt")
	r := ems.UninstallExtensionRequest{
		ChannelID:        "32984",
		ExtensionID:      "13817",
		ExtensionVersion: "4.5.6",
	}

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(uninstallExtensionHandler(http.StatusNoContent, &captured))
	defer server.Close()

	client, err := ems.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	err = client.UninstallExtension(jwt, r)

	t.Run("it should not error", func(t *testing.T) {
		require.NoError(t, err)
	})

	t.Run("it should setup the method and url correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodDelete, req.Method)
		assert.Equal(t, "/channels/32984/extensions/13817/4.5.6", req.URL.String())
	})

	t.Run("it should set the correct header values", func(t *testing.T) {
		assert.Equal(t, jwt.String(), captured.Request.Header.Get("Twitch-Authorization"))
	})

	t.Run("it should have an empty body", func(t *testing.T) {
		assert.Empty(t, captured.Body)
	})
}

func TestUninstallExtension_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(uninstallExtensionHandler(http.StatusForbidden, nil))
	defer server.Close()

	client, err := ems.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	err = client.UninstallExtension("", ems.UninstallExtensionRequest{})

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "uninstall extension unsuccessful [403 Forbidden]", err.Error())
	})
}
