package gql

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"

	"code.justin.tv/gds/gds/extensions/ems/documents"

	"code.justin.tv/extensions/eastwatch/internal/metrics"

	"code.justin.tv/extensions/eastwatch/internal/models/token"
)

//go:generate counterfeiter . Client

// Client defines the GraphQL edge client
type Client interface {

	// GetCurrentUser is a query to get the current user details for the provided user token
	GetCurrentUser(token.OAuth) (User, error)

	// ExtensionsForChannelCurrentUser is a query to get the installed extensions for a given
	// channel as visible to the provided user token
	GetExtensionsForChannelCurrentUser(t token.OAuth, channelID string) (UserChannel, error)

	// CreateExtensionClient is a mutation that creates an extension client id
	CreateExtensionClient(t token.OAuth, extensionName, redirectURI string) (ExtensionClient, error)

	// SaveExtensionManifest updates the manifest for an extension
	SaveExtensionManifest(t token.OAuth, input ExtensionManifestInput) (ExtensionManifest, error)

	// SaveDiscoveryData updates the discovery data for an extension
	SaveDiscoveryData(t token.OAuth, input ExtensionDiscoveryManifestInput) (ExtensionManifest, error)

	// InstallExtension installs an extension onto a channel
	InstallExtension(t token.OAuth, manifestID, channelID string) (InstalledExtension, error)

	// CreatePanel creates a panel on a channel
	CreatePanel(t token.OAuth, channelID string, pt PanelType) (Panel, error)

	// ApplyExtensionActivations will apply the provided extension activations on a channel
	ApplyExtensionActivations(t token.OAuth, channelID string, activations ...ExtensionActivationsInput) ([]InstalledExtension, error)

	// DeletePanel will remove a panel associated to a channel
	DeletePanel(t token.OAuth, panelID string) (Panel, error)

	// UninstallExtension will uninstall an extension from a channel
	UninstallExtension(t token.OAuth, installationID string) (string, error)

	// DeleteExtension will soft delete an extension
	DeleteExtension(t token.OAuth, extensionID string) error

	// GetExtensionCategories gets all the extension categories and extensions in the categories
	GetExtensionCategories(t token.OAuth, first int, cursor string) ([]*ExtensionCategory, error)

	// GetFeaturedSchedule gets the featured extension discovery carousel with the specific ID
	// Note that this returns a FeaturedCarouselDocument; GraphQL variant of this doesn't return
	// FeaturedScheduleDocument-related data
	GetFeaturedSchedule(t token.OAuth, scheduleID string) (documents.FeaturedCarouselDocument, error)

	// SearchExtensions returns an extensions list based on the search term.
	// - If the user of the OAuth Token has the cartman capabilities of extensions::view_all_extensions,
	//   he'll be able to view extensions in any status except "deleted" and "deprecated";
	// - If the user doesn't has the capability but in the whitelist of some extensions,
	//   he'll be able to view all "released" extensions and those who had added him on the whitelist;
	// - If the user doens't has the capability and is not on any whitelist,
	//   he'll only get released extensions
	SearchExtensions(t token.OAuth, first int, after, search string) ([]documents.ExtensionDocument, int, error)

	// GetExtensionRecommendations returns extension recommendations based on the game
	// Currently the list is hard coded in https://git-aws.internal.justin.tv/gds/gds/blob/master/extensions/ems/discoman/documents/content_matched_map.go
	GetExtensionRecommendations(t token.OAuth, gameID string) ([]documents.ExtensionDocument, error)

	// SetExtensionFeatureFlags set the feature flags on the installed extension
	SetExtensionFeatureFlags(t token.OAuth, input SetExtensionFeatureFlagsInput) (InstalledExtension, error)

	// GetExtensionByID gets an extension document by the id
	GetExtensionByID(t token.OAuth, id string) (ExtensionDocument, error)

	// UpdateBroadcastSettings update the channel broadcast settings (game category in this case) and returns the gameID
	UpdateBroadcastSettings(t token.OAuth, channelID string, gameName string) (string, error)
}

type client struct {
	http    *http.Client
	baseURL *url.URL

	query    map[string]Operation
	mutation map[string]Operation
}

// NewClient creates a client that can make requests to GraphQL
func NewClient(http *http.Client, baseURL string) (Client, error) {
	u, err := url.Parse(baseURL)
	if err != nil {
		return nil, err
	}
	q, err := loadOperations("query")
	if err != nil {
		return nil, err
	}
	m, err := loadOperations("mutation")
	if err != nil {
		return nil, err
	}
	return &client{
		http:     http,
		baseURL:  u,
		query:    q,
		mutation: m,
	}, nil
}

// executeMutation will marshal the operation and the input into a valid graphql call and execute it
func (c *client) executeMutation(t token.OAuth, op Operation, input interface{}) (Body, error) {

	body := gqlBody{Query: op.Contents}
	if input != nil {
		body.Variables = &gqlMutationVariables{Input: input}
	}

	b, err := json.Marshal(body)

	if err != nil {
		return nil, err
	}

	d := metrics.Dimensions{
		DependencyServiceName:   "GraphQL",
		DependencyOperationName: op.Name(),
	}

	ctx := metrics.ContextWithDimensions(context.Background(), d)

	req, err := http.NewRequest(http.MethodPost, c.baseURL.String(), bytes.NewBuffer(b))
	if err != nil {
		return nil, err
	}

	req = req.WithContext(ctx)

	req.Header.Set("Authorization", t.HeaderValue())
	req.Header.Set("Content-Type", "application/json")

	resp, err := c.http.Do(req)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("%s unsuccessful [%s]", op.Name(), resp.Status)
	}

	defer resp.Body.Close()
	return ioutil.ReadAll(resp.Body)
}

// executeQuery will marshal the operation and the input into a valid graphql call and execute it
func (c *client) executeQuery(t token.OAuth, op Operation, variables interface{}) (Body, error) {

	body := gqlBody{Query: op.Contents}
	if variables != nil {
		body.Variables = &variables
	}

	b, err := json.Marshal(body)

	if err != nil {
		return nil, err
	}

	d := metrics.Dimensions{
		DependencyServiceName:   "GraphQL",
		DependencyOperationName: op.Name(),
	}

	ctx := metrics.ContextWithDimensions(context.Background(), d)

	req, err := http.NewRequest(http.MethodPost, c.baseURL.String(), bytes.NewBuffer(b))
	if err != nil {
		return nil, err
	}
	req = req.WithContext(ctx)

	req.Header.Set("Authorization", t.HeaderValue())
	req.Header.Set("Content-Type", "application/json")

	resp, err := c.http.Do(req)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("%s unsuccessful [%s]", op.Name(), resp.Status)
	}

	defer resp.Body.Close()
	return ioutil.ReadAll(resp.Body)
}
