package gql

import (
	"time"

	"code.justin.tv/gds/gds/extensions/ems/documents"
	"code.justin.tv/gds/gds/extensions/ems/protocol"
)

// 	=========================================
//	External models. Usually used on reponses.
//	=========================================

// User are the attributes of a user
type User struct {
	ID                  string               `json:"id"`
	DisplayName         string               `json:"displayName"`
	LastBroadcast       LastBroadcast        `json:"lastBroadcast"`
	InstalledExtensions []InstalledExtension `json:"installedExtensions"`
}

// LastBroacst is the user's last broadcast
type LastBroadcast struct {
	Game *Game `json:"game"`
}

// InstalledExtension describes the relationship between an extension and a channel
type InstalledExtension struct {
	// InstallationID is of the form <extID>:<extVer>:<channelID>
	InstallationID    string            `json:"id"`
	ActivationConfig  ActivationConfig  `json:"activationConfig"`
	Extension         ExtensionVersion  `json:"extension"`
	PermittedFeatures PermittedFeatures `json:"permittedFeatures"`
}

// ActivationConfig describes the activation details for an extension on a channel
type ActivationConfig struct {
	Anchor string    `json:"anchor"`
	Slot   PanelSlot `json:"slot"`
	State  string    `json:"state"`
}

// ExtensionVersion describes the details of a specific version of an extension
type ExtensionVersion struct {
	// ManifestID is of the form <extID>:<extVer>
	ManifestID  string `json:"id"`
	ExtensionID string `json:"clientID"`
	Name        string `json:"name"`
	Version     string `json:"version"`
}

// UserChannel is the response body's data field of the
// ExtensionsForChannelCurrentUser query
type UserChannel struct {
	CurrentUser User    `json:"currentUser"`
	Channel     Channel `json:"user"`
}

// Channel describes the installed extensions on a channel
type Channel struct {
	// ID is synonymous with userID or channelID
	ID                  string               `json:"id"`
	InstalledExtensions []InstalledExtension `json:"installedExtensions"`
}

// ExtensionClient is the global details of an extension
type ExtensionClient struct {
	ExtensionID string    `json:"id"`
	Name        string    `json:"name"`
	RedirectURI string    `json:"redirectURI"`
	CreatedAt   time.Time `json:"createdAt"`
}

// ExtensionManifest is the manifest of an extension
type ExtensionManifest struct {
	ManifestID        string                     `json:"id"`
	Version           string                     `json:"version"`
	Capabilities      ExtensionCapabilities      `json:"capabilities"`
	DiscoveryManifest ExtensionDiscoveryManifest `json:"discoveryManifest"`
}

// ExtensionCapabilitiesInput are the capabilities of an extension
type ExtensionCapabilities struct {
	ConfigurationLocation     string              `json:"configurationLocation"`
	HasBitsSupport            bool                `json:"hasBitsSupport"`
	BitsSupportLevel          string              `json:"bitsSupportLevel"`
	HasChatSupport            bool                `json:"hasChatSupport"`
	RequiredConfiguration     string              `json:"requiredConfiguration"`
	SubscriptionsSupportLevel string              `json:"subscriptionsSupportLevel"`
	Whitelists                ExtensionWhitelists `json:"whitelists"`
	WillRequestIdentityLink   bool                `json:"willRequestIdentityLink"`
}

type ExtensionWhitelists struct {
	Broadcasters []string `json:"broadcasters"`
	ConfigURLs   []string `json:"configURLs"`
	PanelURLs    []string `json:"panelURLs"`
	Testers      []string `json:"testers"`
}

type ExtensionDiscoveryManifest struct {
	AuthorName       string   `json:"authorName"`
	Categories       []string `json:"categories"`
	Description      string   `json:"description"`
	Games            []string `json:"games"`
	Name             string   `json:"name"`
	ScreenshotURLs   []string `json:"screenshotURLs"`
	PrivacyPolicyURL string   `json:"privacyPolicyURL"`
	TermsURL         string   `json:"termsURL"`
	Summary          string   `json:"summary"`
	SupportEmail     string   `json:"supportEmail"`
	ViewerSummary    string   `json:"viewerSummary"`
}

type PermittedFeatures struct {
	CanUseBits                    bool                `json:"canUseBits"`
	CanSendChat                   bool                `json:"canSendChat"`
	CanRetrieveSubscriptionStatus bool                `json:"canRetrieveSubscriptionStatus"`
	DynamicManagement             []DynamicManagement `json:"dynamicManagement"`
}

type DynamicManagement struct {
	Game      Game `json:"game"`
	IsManaged bool `json:"isManaged"`
}

type Game struct {
	ID                       string                     `json:"id"`
	Name                     string                     `json:"name"`
	ExtensionRecommendations []extensionRecommendations `json:"extensionRecommendations"`
}

// Panel is a panel on a channel
type Panel struct {
	ID   string    `json:"id"`
	Type PanelType `json:"type"`
}

// PanelType is the type of a panel
type PanelType string

const (
	// ExtensionPanel is an extension panel type
	ExtensionPanel PanelType = "EXTENSION"
)

func (t PanelType) String() string {
	return string(t)
}

type ExtensionCategory struct {
	ID          string
	Name        string
	Description string
	Extensions  []documents.ExtensionDocument
}

// 	=========================================
//	Internal models. Usually used on inputs.
//	=========================================

type extensionsEdge struct {
	Extension extension `json:"extension"`
}

type extension struct {
	ID               string                    `json:"id"`
	AuthorName       string                    `json:"authorName"`
	ClientID         string                    `json:"clientID"`
	IsBitsEnabled    bool                      `json:"isBitsEnabled"`
	BitsSupportLevel protocol.BitsSupportLevel `json:"bitsSupportLevel"`
	Name             string                    `json:"name"`
	State            string                    `json:"state"`
	Summary          string                    `json:"summary"`
	Version          string                    `json:"version"`
	Description      string                    `json:"description"`
}

type pageInfo struct {
	HasNextPage bool `json:"hasNextPage"`
}
