package gql

import (
	"io/ioutil"
	"os"
	"path/filepath"
	"runtime"
	"strings"
)

// baseDir captures the current directory so that we can reference files via relative paths
var baseDir = func() string {
	_, file, _, _ := runtime.Caller(0)
	return file[:strings.LastIndex(file, "/")]
}()

// Operation defines a graphql operation
type Operation struct {
	Type     string
	FileName string
	Contents string
}

func (o Operation) Name() string {
	n := strings.Replace(o.FileName, ".graphql", "", 1)
	return strings.Replace(n, "_", " ", -1)
}

// loadOperations loads all operations in the sub-directory <baseDir>/<operationType>
func loadOperations(operationType string) (map[string]Operation, error) {
	ops := make(map[string]Operation)
	err := filepath.Walk(
		filepath.Join(baseDir, operationType),
		func(path string, info os.FileInfo, err error) error {

			// break on any previous error
			if err != nil {
				return err
			}

			// only load .graphql files
			if !strings.HasSuffix(path, ".graphql") {
				return nil
			}

			// load contents
			b, err := ioutil.ReadFile(path)
			if err != nil {
				return err
			}

			// key is the filename without the extension
			k := strings.Replace(info.Name(), ".graphql", "", 1)

			ops[k] = Operation{
				Type:     operationType,
				FileName: info.Name(),
				Contents: string(b),
			}

			return nil
		},
	)
	return ops, err
}
