package gql

import (
	"errors"

	"code.justin.tv/extensions/eastwatch/internal/models/token"
)

// ExtensionManifestInput is the manifest of an extension
type ExtensionManifestInput struct {
	ID                string                          `json:"id"`
	Version           string                          `json:"version"`
	AssetManifest     ExtensionAssetManifestInput     `json:"assetManifest"`
	Capabilities      ExtensionCapabilitiesInput      `json:"capabilities"`
	DeveloperManifest ExtensionDeveloperManifestInput `json:"developerManifest"`
	DiscoveryManifest ExtensionDiscoveryManifest      `json:"discoveryManifest"`
	Views             ExtensionViewsInput             `json:"views"`
}

type ExtensionAssetManifestInput struct {
	AssetHash string `json:"assetHash"`
}

// ExtensionCapabilitiesInput are the capabilities of an extension
type ExtensionCapabilitiesInput struct {
	ConfigurationLocation     string                   `json:"configurationLocation"`
	HasBitsSupport            bool                     `json:"hasBitsSupport"`
	BitsSupportLevel          string                   `json:"bitsSupportLevel"`
	HasChatSupport            bool                     `json:"hasChatSupport"`
	RequiredConfiguration     string                   `json:"requiredConfiguration"`
	SubscriptionsSupportLevel *string                  `json:"subscriptionsSupportLevel"`
	Whitelists                ExtensionWhitelistsInput `json:"whitelists"`
	WillRequestIdentityLink   bool                     `json:"willRequestIdentityLink"`
}

// ExtensionWhitelistsInput contains whitelists for restricting extension access and behavior.
type ExtensionWhitelistsInput struct {
	Broadcasters []string `json:"broadcasters"`
	ConfigURLs   []string `json:"configURLs"`
	PanelURLs    []string `json:"panelURLs"`
	Testers      []string `json:"testers"`
}

type ExtensionDeveloperManifestInput struct {
	AuthorEmail    string `json:"authorEmail"`
	TestingBaseURI string `json:"testingBaseURI"`
}

// ExtensionDiscoveryManifestInput are the discovery details of an extension
type ExtensionDiscoveryManifestInput struct {
	ID                string                     `json:"id"`
	Version           string                     `json:"version"`
	DiscoveryManifest ExtensionDiscoveryManifest `json:"discoveryManifest"`
}

// ExtensionViewsInput are the view details of an extension
type ExtensionViewsInput struct {
	Component *ComponentViewInput `json:"component"`
	Config    *ConfigViewInput    `json:"config"`
	//Following input is not required but should be cleaned up with gql vendoring.
	//LiveConfig   *LiveConfigViewInput
	//Mobile       *MobileViewInput
	//VideoOverlay *VideoOverlayInput
	Panel *PanelViewInput `json:"panel"`
}

type ComponentViewInput struct {
	AspectHeight int32 `json:"aspectHeight"`
	AspectWidth  int32 `json:"aspectWidth"`
	AspectRatioX int32 `json:"aspectRatioX"`
	AspectRatioY int32 `json:"aspectRatioY"`
	HasAutoscale bool  `json:"hasAutoscale"`
	ScalePixels  int32 `json:"scalePixels"`
	TargetHeight int32 `json:"targetHeight"`
	HasZoom      bool  `json:"hasZoom"`
	//Size         float64 `json:"size"`
	ViewerPath string `json:"viewerPath"`
	ZoomPixels int32  `json:"zoomPixels"`
}

// ConfigViewInput are the details for the config view
type ConfigViewInput struct {
	ViewerPath string `json:"viewerPath"`
}

// PanelViewInput are the details for the panel view
type PanelViewInput struct {
	Height     int    `json:"height"`
	ViewerPath string `json:"viewerPath"`
}

func (c *client) SaveExtensionManifest(t token.OAuth, input ExtensionManifestInput) (ExtensionManifest, error) {
	var body struct {
		SaveExtensionManifest struct {
			Error    string            `json:"error"`
			Manifest ExtensionManifest `json:"manifest"`
		} `json:"saveExtensionManifest"`
	}

	op, found := c.mutation["save_extension_manifest"]
	if !found {
		return ExtensionManifest{}, ErrOperationNotFound
	}

	b, err := c.executeMutation(t, op, input)
	if err != nil {
		return ExtensionManifest{}, err
	}

	err = b.Unmarshal(&body)
	if err != nil {
		return ExtensionManifest{}, err
	}

	// domain error returned as part of the operation
	if body.SaveExtensionManifest.Error != "" {
		return ExtensionManifest{}, errors.New(body.SaveExtensionManifest.Error)
	}

	return body.SaveExtensionManifest.Manifest, nil
}
