package owl_test

import (
	"context"
	"errors"
	"net/url"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/owl"
	"code.justin.tv/extensions/eastwatch/internal/api/owl/owlfakes"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

var (
	fakeUserID = "123456789"
	fakeExtID  = "fake-extension_id"
	owlClient  = new(owlfakes.FakeOwlClient)
)

func TestSuccessfulDeleteClient(t *testing.T) {
	owlClient.DeleteClientReturns(nil)
	client := owl.NewClient(owlClient)

	err := client.DeleteClient(fakeUserID, fakeExtID)

	require.Equal(t, owlClient.DeleteClientCallCount(), 1, "expected DeleteClient to only get called once")
	ctx, extID, params, body := owlClient.DeleteClientArgsForCall(0)

	t.Run("it should not error", func(t *testing.T) {
		assert.NoError(t, err)
	})

	t.Run("it should call the owl client with an empty context", func(t *testing.T) {
		assert.Equal(t, context.Background(), ctx)
	})

	t.Run("it should set the correct parameters", func(t *testing.T) {
		p := url.Values{"user_id": {fakeUserID}}
		assert.Equal(t, p, params)
	})

	t.Run("it should receive the appropriate extension ID", func(t *testing.T) {
		assert.Equal(t, extID, fakeExtID)
	})

	t.Run("it should have an empty body", func(t *testing.T) {
		assert.Nil(t, body)
	})
}

func TestVerifyDeleteClientReturnsError(t *testing.T) {
	fakeErr := errors.New("invalid client id")

	owlClient.DeleteClientReturns(fakeErr)
	client := owl.NewClient(owlClient)

	err := client.DeleteClient(fakeUserID, fakeExtID)

	require.Equal(t, owlClient.DeleteClientCallCount(), 2, "expected DeleteClient to only get called once")
	t.Run("it should return an error", func(t *testing.T) {
		assert.Error(t, err)
	})

	t.Run("it should return the appropriate error", func(t *testing.T) {
		assert.Equal(t, fakeErr.Error(), err.Error())
	})
}
