package owl_test

import (
	"context"
	"errors"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/owl"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	owl2 "code.justin.tv/web/owl/client"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestGetAppAccessToken_Successful(t *testing.T) {
	fakeAccessToken, err := token.NewOAuth("fake-access-token")
	require.NoError(t, err)
	fakeExpiration := 1234567890
	fakeClientID := "fake-client-id"
	fakeClientSecret := "fake-client-secret"

	resp := owl2.ExchangeResponse{
		AccessToken: fakeAccessToken.String(),
		ExpiresIn:   fakeExpiration,
	}

	owlClient.ExchangeClientCredentialsReturns(&resp, nil)

	client := owl.NewClient(owlClient)

	actual, err := client.GetAppAccessToken(fakeClientID, fakeClientSecret)
	require.NoError(t, err)

	require.Equal(t, owlClient.ExchangeClientCredentialsCallCount(), 1)
	ctx, clientID, secret, scope, body := owlClient.ExchangeClientCredentialsArgsForCall(0)

	t.Run("it should return an app access token", func(t *testing.T) {
		require.Equal(t, fakeAccessToken, actual)
	})

	t.Run("it should call the owl client with an empty context", func(t *testing.T) {
		assert.Equal(t, context.Background(), ctx)
	})

	t.Run("it should have the right parameters", func(t *testing.T) {
		assert.Equal(t, clientID, fakeClientID)
		assert.Equal(t, secret, fakeClientSecret)
	})

	t.Run("it should have an empty body", func(t *testing.T) {
		assert.Nil(t, body)
	})

	t.Run("it should have an empty scope", func(t *testing.T) {
		assert.Nil(t, scope)
	})

	t.Run("it should return the right values", func(t *testing.T) {
		assert.Equal(t, fakeAccessToken, actual)
	})
}

func TestGetAppAccessToken_Unsuccessful(t *testing.T) {
	fakeErr := errors.New("invalid client id")

	owlClient.ExchangeClientCredentialsReturns(nil, fakeErr)
	client := owl.NewClient(owlClient)

	_, err := client.GetAppAccessToken("", "")

	require.Equal(t, owlClient.ExchangeClientCredentialsCallCount(), 2)

	t.Run("it should throw an error", func(t *testing.T) {
		require.Error(t, err)
		assert.Equal(t, "GetAppAccessToken failed: "+fakeErr.Error(), err.Error())
	})
}
