package owl_test

import (
	"context"
	"errors"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/owl"
	"code.justin.tv/extensions/eastwatch/internal/api/owl/owlfakes"
	owlclient "code.justin.tv/web/owl/client"
	"code.justin.tv/web/owl/oauth2"
	"github.com/stretchr/testify/assert"
)

var (
	fakeAppID       = "fake-app-id"
	fakeAppName     = "fake-app-name"
	fakeOauthClient = []*oauth2.Client{
		{
			ClientID: fakeAppID,
			Name:     fakeAppName,
		},
	}
)

func TestSuccessfulGetClients(t *testing.T) {

	fakeOwlClient := new(owlfakes.FakeOwlClient)
	fakeOwlClient.GetClientsReturns(fakeOauthClient, "", nil)
	client := owl.NewClient(fakeOwlClient)

	clients, err := client.GetClients(fakeUserID)

	ctx, cursor, showHidden, isExtension, filters, sortKey, sortOrder, reqOpts := fakeOwlClient.GetClientsArgsForCall(0)

	t.Run("it should not error", func(t *testing.T) {
		assert.Nil(t, err)
	})

	t.Run("it should use an empty context", func(t *testing.T) {
		assert.Equal(t, context.Background(), ctx)
	})

	t.Run("it should use an empty cursor", func(t *testing.T) {
		assert.Equal(t, "", cursor)
	})

	t.Run("it should show hidden clients", func(t *testing.T) {
		assert.Equal(t, true, showHidden)
	})

	t.Run("it should show extensions", func(t *testing.T) {
		assert.Equal(t, true, isExtension)
	})

	t.Run("it should configure the appropriate filter", func(t *testing.T) {
		f := map[owlclient.FilterableColumn]string{
			owlclient.FilterColOwnerID: fakeUserID,
		}
		assert.Equal(t, f, filters)
	})

	t.Run("it should sort by ClientID", func(t *testing.T) {
		assert.Equal(t, owlclient.SortKeyClientID, sortKey)
	})

	t.Run("it should sort in ascending order", func(t *testing.T) {
		assert.Equal(t, owlclient.SortAsc, sortOrder)
	})

	t.Run("it should have no body", func(t *testing.T) {
		assert.Nil(t, reqOpts)
	})

	t.Run("it should return the client", func(t *testing.T) {
		assert.Equal(t, fakeOauthClient, clients)
	})

}

func TestUnsuccessfulGetClients(t *testing.T) {
	fakeErr := errors.New("this is a test error")
	fakeOwlClient := new(owlfakes.FakeOwlClient)
	fakeOwlClient.GetClientsReturns(nil, "", fakeErr)
	client := owl.NewClient(fakeOwlClient)

	_, err := client.GetClients(fakeUserID)

	t.Run("it should return the appropriate test error", func(t *testing.T) {
		assert.Equal(t, fakeErr, err)
	})
}
