package panel_test

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"strconv"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/panel"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func GetChannelPanelsHandler(returnStatusCode int, panels []panel.Panel, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if returnStatusCode != http.StatusOK {
			w.WriteHeader(returnStatusCode)
		}
		result := struct {
			Panels []panel.Panel `json:"panels"`
		}{panels}
		b, err := json.Marshal(result)
		if err != nil {
			panic(err)
		}
		w.Write(b)
	})
	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestGetChannelPanels_Successful(t *testing.T) {
	channelID := 32109

	panels := []panel.Panel{
		{
			ID:           10001,
			UserID:       channelID,
			DisplayOrder: 0,
			Kind:         "extension",
			Data: panel.Data{
				Description: "",
				Slot:        "extension-panel-1",
			},
			HTMLDescription: "",
		},
		{
			ID:           10002,
			UserID:       channelID,
			DisplayOrder: 0,
			Kind:         "extension",
			Data: panel.Data{
				Description: "",
				Slot:        "extension-panel-2",
			},
			HTMLDescription: "",
		},
	}

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(GetChannelPanelsHandler(http.StatusOK, panels, &captured))
	defer server.Close()

	client, err := panel.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	actual, err := client.GetChannelPanels(strconv.Itoa(channelID))
	require.NoError(t, err)

	t.Run("it should setup the method and url correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodGet, req.Method)
		assert.Equal(t, fmt.Sprintf("/v1/users/%d/panels?include_extensions=1", channelID), req.URL.String())
	})

	t.Run("it should return the correct value", func(t *testing.T) {
		assert.Equal(t, panels, actual)
	})
}

func TestGetChannelPanels_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(GetChannelPanelsHandler(http.StatusNotFound, nil, nil))
	defer server.Close()

	client, err := panel.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	_, err = client.GetChannelPanels("0")

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "get channel panels unsuccessful [404 Not Found]", err.Error())
	})
}
