package visage

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"

	"code.justin.tv/extensions/eastwatch/internal/models/token"
)

// AddExtensionRequest is a simplified add extension body required for Eastwatch testing purposes
type AddExtensionRequest struct {
	ExtensionID      string
	ExtensionVersion string
	ExtensionName    string
	AuthorID         string
	AuthorName       string
}

type addExtensionRequest struct {
	ExtensionID      string                             `json:"extension_id"`
	Name             string                             `json:"name"`
	Summary          string                             `json:"summary"`
	Description      string                             `json:"description"`
	ExtensionVersion string                             `json:"extension_version"`
	AuthorName       string                             `json:"author_name"`
	AuthorEmail      string                             `json:"author_email"`
	SupportEmail     string                             `json:"support_email"`
	IconPath         string                             `json:"icon_path"`
	ScreenshotPaths  []string                           `json:"screenshot_paths"`
	TestingAccounts  []string                           `json:"testing_accounts"`
	TestingBaseURI   string                             `json:"testing_base_uri"`
	Views            map[string]addExtensionRequestView `json:"views"`
}

type addExtensionRequestView struct {
	ViewerPath string `json:"viewer_path"`
	Height     int    `json:"height,omitempty"`
}

func buildAddExtension(r AddExtensionRequest) addExtensionRequest {
	return addExtensionRequest{
		ExtensionID:      r.ExtensionID,
		ExtensionVersion: r.ExtensionVersion,
		Name:             r.ExtensionName,
		TestingAccounts:  []string{r.AuthorID},
		AuthorName:       r.AuthorName,
		Summary:          "eastwatch test extension",
		Description:      "extension for testing with eastwatch",
		AuthorEmail:      "dx-discoman-qa@twitch.tv",
		SupportEmail:     "dx-discoman-qa@twitch.tv",
		IconPath:         "https://media.forgecdn.net/avatars/185/170/636825808928162622.png",
		ScreenshotPaths:  []string{"internal/screenshots/test_data.png"},
		TestingBaseURI:   "https://localhost:8080/",
		Views: map[string]addExtensionRequestView{
			"panel": addExtensionRequestView{
				ViewerPath: "panel.html",
				Height:     300,
			},
			"config": addExtensionRequestView{
				ViewerPath: "config.html",
			},
		},
	}
}

func (c *client) AddExtension(t token.OAuth, r AddExtensionRequest) error {
	// POST /v5/extensions
	u, err := url.Parse("v5/extensions")
	if err != nil {
		return err
	}

	body, err := json.Marshal(buildAddExtension(r))
	if err != nil {
		return err
	}

	req, err := c.buildRequest(context.Background(), "AddExtension", http.MethodPost, c.baseURL.ResolveReference(u).String(), bytes.NewBuffer(body))
	if err != nil {
		return err
	}
	req.Header.Set("Authorization", t.HeaderValue())
	req.Header.Set("Content-Type", "application/json")

	resp, err := c.http.Do(req)
	if err != nil {
		return err
	}
	if resp.StatusCode != http.StatusNoContent {
		return fmt.Errorf("add extension unsuccessful [%s]", resp.Status)
	}

	return nil
}
