package visage_test

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/visage"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func addExtensionHandler(returnStatusCode int, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(returnStatusCode)
	})
	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestAddExtension_Successful(t *testing.T) {
	oauth, err := token.NewOAuth("fake-oauth-token")
	require.NoError(t, err)
	r := visage.AddExtensionRequest{
		ExtensionID:      "ja1bta173mqtkhnoayun3nyofa8wpg",
		ExtensionVersion: "0.0.1",
		ExtensionName:    "eastwatch-test-extension",
		AuthorID:         "277015761",
		AuthorName:       "dxdiscomanqa3",
	}

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(addExtensionHandler(http.StatusNoContent, &captured))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	err = client.AddExtension(oauth, r)
	require.NoError(t, err)

	t.Run("it should setup the method and url correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodPost, req.Method)
		assert.Equal(t, "/v5/extensions", req.URL.String())
	})

	t.Run("it should set the correct header values", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, "OAuth fake-oauth-token", req.Header.Get("Authorization"))
		assert.Equal(t, "application/json", req.Header.Get("Content-Type"))
	})

	t.Run("it should setup the body created correctly", func(t *testing.T) {
		var actual, expected map[string]interface{}
		require.NoError(t, testutil.UnmarshalTestData("add_extension_request.json", &expected))
		require.NoError(t, json.Unmarshal(captured.Body, &actual))
		assert.Equal(t, expected, actual)
	})
}

func TestAddExtension_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(addExtensionHandler(http.StatusInternalServerError, nil))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	err = client.AddExtension("", visage.AddExtensionRequest{})

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "add extension unsuccessful [500 Internal Server Error]", err.Error())
	})
}
