package visage_test

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/visage"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"code.justin.tv/gds/gds/extensions/ems/documents"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func addFeaturedCarouselEntryHandler(returnStatusCode int, doc *documents.FeaturedCarouselEntryDocument, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if returnStatusCode != http.StatusOK {
			w.WriteHeader(returnStatusCode)
		}
		b, err := json.Marshal(doc)
		if err != nil {
			panic(err)
		}
		w.Write(b)
	})

	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestAddFeaturedCarouselEntry_Successful(t *testing.T) {
	oauth, err := token.NewOAuth("fake-oauth-token")
	require.NoError(t, err)

	carouselid := "fake-caorusel-id"
	order := float32(5)
	content := documents.FeaturedCarouselEntryContent{
		ClickThrough: "test-clickthrough-url",
		ImageURL:     "test-image-url",
		Title:        "Test Title",
	}

	entry := documents.AddFeaturedCarouselEntryRequest{
		CarouselID: &carouselid,
		Content:    &content,
		Order:      &order,
	}

	doc := documents.FeaturedCarouselEntryDocument{
		CarouselID: carouselid,
		Content:    content,
		Order:      order,
	}

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(addFeaturedCarouselEntryHandler(http.StatusOK, &doc, &captured))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	actual, err := client.AddFeaturedCarouselEntry(oauth, entry)
	require.NoError(t, err)

	t.Run("it should return the correct value", func(t *testing.T) {
		require.Equal(t, &doc, actual)
	})

	t.Run("it should set up the method and URL correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodPost, req.Method)
		assert.Equal(t, "/v5/featured_carousel_entries", req.URL.String())
	})

	t.Run("it should set the correct header values", func(t *testing.T) {
		assert.Equal(t, "OAuth fake-oauth-token", captured.Request.Header.Get("Authorization"))
	})

}

func TestAddFeaturedCarouselEntry_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(addFeaturedCarouselEntryHandler(http.StatusUnauthorized, nil, nil))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	_, err = client.AddFeaturedCarouselEntry("", documents.AddFeaturedCarouselEntryRequest{})

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "add featured carousel entry unsuccessful [401 Unauthorized]", err.Error())
	})
}
