package visage

import (
	"context"
	"io"
	"net/http"
	"net/url"

	"code.justin.tv/extensions/eastwatch/internal/metrics"

	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/gds/gds/extensions/ems/documents"
)

//go:generate counterfeiter . Client

// Client exposes behavior available in Visage
type Client interface {
	// CreateExtensionID will create an extension id
	CreateExtensionID(t token.OAuth, extensionName string) (extensionID string, err error)

	// AddExtension will create an extension
	AddExtension(token.OAuth, AddExtensionRequest) error

	// TransitionExtensionState will request to transition an extension to a different state
	TransitionExtensionState(token.OAuth, TransitionExtensionStateRequest) error

	// GetExtensionsVersion will get the details of a specific version of an extension
	GetExtensionVersion(t token.OAuth, extensionID, extensionVersion string) (documents.ExtensionDocument, error)

	// GetExtensionsByChannelID will get all of the extensions on a channel visible to the user
	GetExtensionsByChannelID(t token.OAuth, channelID string) (documents.InstalledExtensionsDocument, error)

	// GetMobileExtensionsByChannelID will get all of the extensions on a channel visible to the user
	// on a mobile device
	GetMobileExtensionsByChannelID(t token.OAuth, channelID string) (documents.InstalledExtensionsDocument, error)

	// GetExtensionByID will get a specific extension at the specified version
	GetExtensionByID(t token.OAuth, extensionID, extensionVersion string) (documents.ExtensionDocument, error)

	// AddCategory creates a new extension category
	AddCategory(t token.OAuth, r documents.AddCategoryRequest) (*documents.CategoryDocument, error)

	// DeleteCategory deletes an extension category
	DeleteCategory(t token.OAuth, categoryID string) (documents.CategoryDocument, error)

	// EditCategoryTranslation modifies a given category's translation for a given language
	EditCategoryTranslation(t token.OAuth, categoryID string, language string, request documents.EditCategoryTranslationRequest) (*documents.CategoryDocument, error)

	// UpdateCategory modifies an existing extension
	UpdateCategory(t token.OAuth, categoryID string, r documents.AddCategoryRequest) (*documents.CategoryDocument, error)

	// AddExtensionToCategory adds an extension to a category
	AddExtensionToCategory(t token.OAuth, categoryID string, extensionID string, r *documents.AddExtensionToCategoryRequest) (*documents.ExtensionCategoryMembershipDocument, error)

	// GetFeaturedSchedules gets all the featured schedules for extensions carousels
	GetFeaturedSchedules(t token.OAuth, offset int, limit int) (*documents.FeaturedSchedulesDocument, error)

	// AddFeaturedCarouselEntry adds an entry to a given featured carousel
	AddFeaturedCarouselEntry(t token.OAuth, r documents.AddFeaturedCarouselEntryRequest) (*documents.FeaturedCarouselEntryDocument, error)

	// UpdateFeaturedCarouselEntry updates an entry to a given featured carouosel
	UpdateFeaturedCarouselEntry(t token.OAuth, r documents.AddFeaturedCarouselEntryRequest) (*documents.FeaturedCarouselEntryDocument, error)

	// DeleteFeaturedCarousel removes a featured carousel
	DeleteFeaturedCarousel(t token.OAuth, categoryID string) (*documents.FeaturedCarouselDocument, error)

	// DeleteFeaturedCarouselEntry removes a featured carousel entry
	DeleteFeaturedCarouselEntry(t token.OAuth, entryID string) (*documents.FeaturedCarouselEntryDocument, error)

	// RemoveExtensionFromCategory removes an extension from a category
	RemoveExtensionFromCategory(t token.OAuth, categoryID string, extensionID string) (*documents.ExtensionCategoryMembershipDocument, error)

	// OrderCategory changes the order of extensions in a given category
	OrderCategory(t token.OAuth, categoryID string, r documents.OrderCategoryRequest) (*documents.CategoryDocument, error)

	// GetFeaturedCarousels gets featured carousels based on a limit and an offset
	GetFeaturedCarousels(t token.OAuth, offset int, limit int) (*documents.FeaturedCarouselsDocument, error)

	// AddFeaturedCarousel creates a new Featured Carousel
	AddFeaturedCarousel(t token.OAuth, r documents.AddFeaturedCarouselRequest) (*documents.FeaturedCarouselDocument, error)
}

type client struct {
	http        *http.Client
	baseURL     *url.URL
	ServiceName string
}

// NewClient creates a client that can make requests to Visage
func NewClient(http *http.Client, baseURL string) (Client, error) {
	u, err := url.Parse(baseURL)
	if err != nil {
		return nil, err
	}
	return &client{
		http:        http,
		baseURL:     u,
		ServiceName: "Visage",
	}, nil
}

func (c *client) buildRequest(ctx context.Context, operationName string, method string, url string, body io.Reader) (*http.Request, error) {
	d := metrics.Dimensions{
		DependencyServiceName:   c.ServiceName,
		DependencyOperationName: operationName,
	}

	ctx = metrics.ContextWithDimensions(ctx, d)

	req, err := http.NewRequest(method, url, body)
	if err != nil {
		return nil, err
	}

	return req.WithContext(ctx), nil

}
