package visage

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"

	"code.justin.tv/extensions/eastwatch/internal/models/token"
)

// CreateExtensionIDRequestBody is the body required to make the request to Kraken to create
// an extension id
type CreateExtensionIDRequestBody struct {
	Name          string `json:"name"`
	RedirectURI   string `json:"redirect_uri"`
	CategoryID    string `json:"category_id"`
	CategoryOther string `json:"category_other"`
}

// CreateExtensionIDResponseBody is the body response from visage
type CreateExtensionIDResponseBody struct {
	ExtensionID string `json:"id"`
}

func buildCreateExtensionIDRequest(name string) CreateExtensionIDRequestBody {
	return CreateExtensionIDRequestBody{
		Name: name,

		// Hardcoded values for purposes of Eastwatch
		RedirectURI:   "http://localhost:8000",
		CategoryID:    "9", // Other
		CategoryOther: "Eastwatch",
	}
}

func (c *client) CreateExtensionID(t token.OAuth, extensionName string) (string, error) {
	u, err := url.Parse("kraken/apps")
	if err != nil {
		return "", err
	}

	body, err := json.Marshal(buildCreateExtensionIDRequest(extensionName))
	if err != nil {
		return "", err
	}

	req, err := c.buildRequest(context.Background(), "CreateExtensionID", http.MethodPost, c.baseURL.ResolveReference(u).String(), bytes.NewBuffer(body))
	if err != nil {
		return "", err
	}
	req.Header.Set("Accept", "application/vnd.twitchtv.v5+json")
	req.Header.Set("Authorization", t.HeaderValue())
	req.Header.Set("Content-Type", "application/json")

	resp, err := c.http.Do(req)
	if err != nil {
		return "", err
	}
	if resp.StatusCode != http.StatusOK {
		return "", fmt.Errorf("create extension id unsuccessful [%s]", resp.Status)
	}

	defer resp.Body.Close()
	var data CreateExtensionIDResponseBody
	err = json.NewDecoder(resp.Body).Decode(&data)
	if err != nil {
		return "", err
	}

	return data.ExtensionID, nil
}
