package visage_test

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/visage"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func createExtensionIDHandler(returnStatusCode int, extensionID string, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if returnStatusCode != http.StatusOK {
			w.WriteHeader(returnStatusCode)
			return
		}

		out, err := json.Marshal(visage.CreateExtensionIDResponseBody{
			ExtensionID: extensionID,
		})
		if err != nil {
			panic(err)
		}
		w.Write(out)
	})
	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestCreateExtensionID_Successful(t *testing.T) {
	extensionID := "extID123"
	oauth, err := token.NewOAuth("fake-oauth-token")
	require.NoError(t, err)
	extensionName := "eastwatch-test-extension"

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(createExtensionIDHandler(http.StatusOK, extensionID, &captured))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	actual, err := client.CreateExtensionID(oauth, extensionName)
	require.NoError(t, err)

	t.Run("it should return the correct extension id", func(t *testing.T) {
		require.Equal(t, extensionID, actual)
	})

	t.Run("it should setup the method and url correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodPost, req.Method)
		assert.Equal(t, "/kraken/apps", req.URL.String())
	})

	t.Run("it should set the correct header values", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, "application/vnd.twitchtv.v5+json", req.Header.Get("Accept"))
		assert.Equal(t, "OAuth fake-oauth-token", req.Header.Get("Authorization"))
		assert.Equal(t, "application/json", req.Header.Get("Content-Type"))
	})

	t.Run("it should setup the body created correctly", func(t *testing.T) {
		var actual, expected map[string]interface{}
		require.NoError(t, testutil.UnmarshalTestData("create_extension_id_request.json", &expected))
		require.NoError(t, json.Unmarshal(captured.Body, &actual))
		assert.Equal(t, expected, actual)
	})
}

func TestCreateExtensionID_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(createExtensionIDHandler(http.StatusInternalServerError, "", nil))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	_, err = client.CreateExtensionID("", "")

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "create extension id unsuccessful [500 Internal Server Error]", err.Error())
	})
}
