package visage_test

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/visage"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"code.justin.tv/gds/gds/extensions/ems/documents"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func deleteCategoryHandler(returnStatusCode int, doc documents.CategoryDocument, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if returnStatusCode != http.StatusOK {
			w.WriteHeader(returnStatusCode)
		}
		b, err := json.Marshal(doc)
		if err != nil {
			panic(err)
		}
		w.Write(b)
	})
	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestDeleteCategory_Successful(t *testing.T) {
	oauth, err := token.NewOAuth("fake-oauth-token")
	require.NoError(t, err)
	categoryID := "fake-category-id"

	doc := documents.CategoryDocument{
		ID: categoryID,
	}

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(deleteCategoryHandler(http.StatusOK, doc, &captured))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	actual, err := client.DeleteCategory(oauth, categoryID)
	require.NoError(t, err)

	t.Run("it should return the correct value", func(t *testing.T) {
		require.Equal(t, doc, actual)
	})

	t.Run("it should setup the method and url correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodDelete, req.Method)
		assert.Equal(t, "/v5/categories/fake-category-id", req.URL.String())
	})

	t.Run("it should set the correct header values", func(t *testing.T) {
		assert.Equal(t, "OAuth fake-oauth-token", captured.Request.Header.Get("Authorization"))
	})
}

func TestDeleteCategory_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(deleteCategoryHandler(http.StatusUnauthorized, documents.CategoryDocument{}, nil))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	_, err = client.DeleteCategory("", "")

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "delete category unsuccessful [401 Unauthorized]", err.Error())
	})
}
