package visage_test

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/extensions/eastwatch/internal/api/visage"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"github.com/stretchr/testify/require"

	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"code.justin.tv/gds/gds/extensions/ems/documents"
)

func getExtensionByIDHandler(returnStatusCode int, doc documents.ExtensionDocument, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if returnStatusCode != http.StatusOK {
			w.WriteHeader(returnStatusCode)
		}
		b, err := json.Marshal(doc)
		if err != nil {
			panic(err)
		}
		w.Write(b)
	})
	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestGetExtensionByID_Successful(t *testing.T) {
	oauth, err := token.NewOAuth("fake-oauth-token")
	require.NoError(t, err)
	extensionID := "123456"
	extensionVersion := "2.1.3"

	doc := documents.ExtensionDocument{
		ID:      extensionID,
		Version: extensionVersion,
	}

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(getExtensionByIDHandler(http.StatusOK, doc, &captured))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	actual, err := client.GetExtensionByID(oauth, extensionID, extensionVersion)
	require.NoError(t, err)

	t.Run("it should return the correct value", func(t *testing.T) {
		require.Equal(t, doc, actual)
	})

	t.Run("it should setup the method and url correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodGet, req.Method)
		assert.Equal(t, "/v5/extensions/123456/2.1.3", req.URL.String())
	})

	t.Run("it should set the correct header values", func(t *testing.T) {
		assert.Equal(t, "OAuth fake-oauth-token", captured.Request.Header.Get("Authorization"))
	})

}

func TestGetExtensionByID_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(getExtensionByIDHandler(http.StatusUnauthorized, documents.ExtensionDocument{}, nil))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	_, err = client.GetExtensionByID("", "", "")

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "get extension by id unsuccessful [401 Unauthorized]", err.Error())
	})
}
