package visage_test

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/visage"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"code.justin.tv/gds/gds/extensions/ems/documents"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func getExtensionVersionHandler(returnStatusCode int, doc documents.ExtensionDocument, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if returnStatusCode != http.StatusOK {
			w.WriteHeader(returnStatusCode)
		}
		b, err := json.Marshal(doc)
		if err != nil {
			panic(err)
		}
		w.Write(b)
	})
	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestGetExtensionVersion_Successful(t *testing.T) {
	oauth, err := token.NewOAuth("fake-oauth-token")
	require.NoError(t, err)
	extensionsID := "46253"
	extensionsVersion := "2.3.1"
	doc := documents.ExtensionDocument{
		ID:      extensionsID,
		Version: extensionsVersion,
		Name:    "some-extension",
		Summary: "for unit testing",
	}

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(getExtensionVersionHandler(http.StatusOK, doc, &captured))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	actual, err := client.GetExtensionVersion(oauth, extensionsID, extensionsVersion)
	require.NoError(t, err)

	t.Run("it should return the correct value", func(t *testing.T) {
		require.Equal(t, doc, actual)
	})

	t.Run("it should setup the method and url correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodGet, req.Method)
		assert.Equal(t, "/v5/extensions/46253/2.3.1", req.URL.String())
	})

	t.Run("it should set the correct header values", func(t *testing.T) {
		assert.Equal(t, "OAuth fake-oauth-token", captured.Request.Header.Get("Authorization"))
	})
}

func TestGetExtensionVersion_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(getExtensionVersionHandler(http.StatusNotFound, documents.ExtensionDocument{}, nil))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	_, err = client.GetExtensionVersion("", "", "")

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "get extension version unsuccessful [404 Not Found]", err.Error())
	})
}
