package visage_test

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"github.com/stretchr/testify/assert"

	"code.justin.tv/extensions/eastwatch/internal/api/visage"
	"github.com/stretchr/testify/require"

	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"code.justin.tv/gds/gds/extensions/ems/documents"
)

func getFeaturedSchedulesHandler(returnStatusCode int, doc *documents.FeaturedSchedulesDocument, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if returnStatusCode != http.StatusOK {
			w.WriteHeader(returnStatusCode)
		}
		b, err := json.Marshal(doc)
		if err != nil {
			panic(err)
		}
		w.Write(b)
	})
	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestGetFeaturedSchedules_Successful(t *testing.T) {
	oauth, err := token.NewOAuth("fake-oauth-token")
	require.NoError(t, err)

	limit := 20
	offset := 0

	doc := documents.FeaturedSchedulesDocument{
		Count: 1,
		Schedules: []*documents.FeaturedScheduleDocument{
			{
				ID:          "fake-schedule-id",
				Description: "fake-carousel-desc",
				CurrentCarousel: &documents.FeaturedCarouselDocument{
					ID: "fake-carousel-id",
				},
			},
		},
	}

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(getFeaturedSchedulesHandler(http.StatusOK, &doc, &captured))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	actual, err := client.GetFeaturedSchedules(oauth, offset, limit)
	require.NoError(t, err)

	t.Run("it should return the correct values", func(t *testing.T) {
		assert.Equal(t, &doc, actual)
	})

	t.Run("it should set up the method and url correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodGet, req.Method)
		assert.Equal(t, fmt.Sprintf("/v5/featured_schedules?limit=%d&offset=%d", limit, offset), req.URL.String())
	})

	t.Run("it should set up the correct header values", func(t *testing.T) {
		assert.Equal(t, "OAuth fake-oauth-token", captured.Request.Header.Get("Authorization"))
	})
}

func TestGetFeaturedSchedules_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(getFeaturedSchedulesHandler(http.StatusUnauthorized, nil, nil))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	_, err = client.GetFeaturedSchedules("", 0, 0)

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "get featured schedules unsuccessful [401 Unauthorized]", err.Error())
	})
}
