package visage_test

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/extensions/eastwatch/internal/api/visage"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"code.justin.tv/gds/gds/extensions/ems/data"
	"code.justin.tv/gds/gds/extensions/ems/documents"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func transitionExtensionStateHandler(returnStatusCode int, c *testutil.CapturedRequestDetails) http.HandlerFunc {
	h := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(returnStatusCode)
	})
	return testutil.CaptureRequestBodyMiddleware(h, c)
}

func TestTransitionExtensionState_Successful(t *testing.T) {
	oauth, err := token.NewOAuth("fake-oauth-token")
	require.NoError(t, err)
	r := visage.TransitionExtensionStateRequest{
		ExtensionID:      "28475",
		ExtensionVersion: "5.6.2",
		State:            data.InReview,
	}

	var captured testutil.CapturedRequestDetails
	server := httptest.NewServer(transitionExtensionStateHandler(http.StatusNoContent, &captured))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	err = client.TransitionExtensionState(oauth, r)

	t.Run("it should not error", func(t *testing.T) {
		require.NoError(t, err)
	})

	t.Run("it should setup the method and url correctly", func(t *testing.T) {
		req := captured.Request
		assert.Equal(t, http.MethodPut, req.Method)
		assert.Equal(t, "/v5/extensions/28475/5.6.2", req.URL.String())
	})

	t.Run("it should set the correct header values", func(t *testing.T) {
		assert.Equal(t, "OAuth fake-oauth-token", captured.Request.Header.Get("Authorization"))
		assert.Equal(t, "application/json", captured.Request.Header.Get("Content-Type"))
	})

	t.Run("it should the correct body with the state value", func(t *testing.T) {
		var actual, expected map[string]interface{}
		require.NoError(t, testutil.UnmarshalTestData("transition_extension_state_request.json", &expected))
		require.NoError(t, json.Unmarshal(captured.Body, &actual))
		assert.Equal(t, expected, actual)
	})
}

func TestTransitionExtensionState_Unsuccessful(t *testing.T) {
	server := httptest.NewServer(getExtensionsByChannelIDHandler(http.StatusBadRequest, documents.InstalledExtensionsDocument{}, nil))
	defer server.Close()

	client, err := visage.NewClient(server.Client(), server.URL)
	require.NoError(t, err)

	err = client.TransitionExtensionState("", visage.TransitionExtensionStateRequest{})

	t.Run("it should error", func(t *testing.T) {
		require.Error(t, err)
		require.Equal(t, "transition extension state unsuccessful [400 Bad Request]", err.Error())
	})
}
