package metrics

import (
	"context"
	"encoding/json"
	"errors"
	"os"
	"time"

	cloudwatch "code.justin.tv/amzn/TwitchTelemetryCloudWatchMetricsSender"

	loggers "code.justin.tv/amzn/TwitchLoggingCommonLoggers"
	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/extensions/eastwatch/internal/fultonlibs/FultonGoLangSimpleBootstrap/simplebootstrap"
)

// actual key is the address of this var, which is unique in program space
var dimensionsKey int

type Dimensions struct {
	DependencyServiceName   string
	DependencyOperationName string
}

func ContextWithDimensions(ctx context.Context, dimensions Dimensions) context.Context {
	return context.WithValue(ctx, &dimensionsKey, dimensions)
}

func GetDimensions(ctx context.Context) (Dimensions, error) {
	var dimensions Dimensions
	if ctx.Value(&dimensionsKey) != nil {
		return ctx.Value(&dimensionsKey).(Dimensions), nil
	}
	return dimensions, errors.New("missing dimensions in context")
}

func InitBootStrap(serviceName string) (*simplebootstrap.FultonBootstrap, error) {
	pid := identifier.ProcessIdentifier{
		Service: serviceName,
		Region:  "us-west-2",
		Stage:   "staging",
	}

	writer := &telemetry.SampleReportingWriter{
		Writer:                 os.Stdout,
		BytesWrittenMetricName: telemetry.MetricStdoutBytesWritten,
		DurationMetricName:     telemetry.MetricStdoutWaitTime,
	}

	encoder := json.NewEncoder(writer)
	jsonLogger := &loggers.JSONLogger{Dest: encoder, OnError: nil}

	sender := cloudwatch.NewUnbuffered(&pid, jsonLogger)
	sampleObserver := telemetry.NewBufferedAggregator(30*time.Second, 100000, time.Minute, sender, jsonLogger)

	customMembers := simplebootstrap.FultonCustomMembers{
		Logger:           jsonLogger,
		SampleObserver:   sampleObserver,
		AutoFlushMetrics: false,
	}

	simpleBootstrap, err := simplebootstrap.CommonSimpleBootstrap(&pid, customMembers)
	if err != nil {
		return nil, err
	}

	if simpleBootstrap == nil {
		return nil, errors.New("cannot use empty bootstrap")
	}

	writer.Reporter = &simpleBootstrap.SampleReporter
	return simpleBootstrap, nil

}
