package testutil

import (
	"fmt"
	"net/http"
	"net/http/httputil"
	"sync"
)

var _ http.RoundTripper = &RoundTripRecorder{}

// RoundTripRecorder will log all requests and response/errors in and out and make them
// available for inspection
type RoundTripRecorder struct {
	wrapped   http.RoundTripper
	observers []func(RoundTripRecord)

	m       sync.RWMutex
	records []RoundTripRecord
}

// RoundTripRecord collects the request and response/error from a single http round trip
type RoundTripRecord struct {
	Request      *http.Request
	Response     *http.Response
	RequestDump  string
	ResponseDump string
	Error        error
}

// NewRoundTripRecorder creates a new round tripper recorder
func NewRoundTripRecorder(rt http.RoundTripper, observers ...func(RoundTripRecord)) *RoundTripRecorder {
	return &RoundTripRecorder{
		wrapped:   rt,
		observers: observers,
	}
}

func (c *RoundTripRecorder) printDump(b []byte, err error) string {
	if err != nil {
		return fmt.Sprint(err)
	}
	return string(b)
}

// RoundTrip will call the underlying round tripper but record the responses and requests/errors
func (c *RoundTripRecorder) RoundTrip(req *http.Request) (*http.Response, error) {

	rec := RoundTripRecord{
		Request:     req,
		RequestDump: c.printDump(httputil.DumpRequestOut(req, true)),
	}

	rec.Response, rec.Error = c.wrapped.RoundTrip(req)

	if rec.Error == nil && rec.Response != nil {
		rec.ResponseDump = c.printDump(httputil.DumpResponse(rec.Response, true))
	}

	c.m.Lock()
	c.records = append(c.records, rec)
	c.m.Unlock()

	for _, o := range c.observers {
		o(rec)
	}

	return rec.Response, rec.Error
}

// RoundTripRecords are all of the round trip records recorded
func (c *RoundTripRecorder) RoundTripRecords() []RoundTripRecord {
	c.m.RLock()
	defer c.m.RUnlock()
	return c.records
}
