package testutil

import (
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"path/filepath"
)

// UnmarshalTestData reads in the file located at path `testdata/<file>` relative to the caller
// and json unmarshals it into the specified value
func UnmarshalTestData(file string, v interface{}) error {
	b, err := ioutil.ReadFile(filepath.Join("testdata", filepath.Clean(file)))
	if err != nil {
		return err
	}
	return json.Unmarshal(b, v)
}

// UnmarshalReaderJSON will read all data from the io.Reader as a string and then unmarshal
// that as json into the provided value `v`, if the json unmarshal fails then the error returned
// will include the original bytes read from the io.Reader.
//
// This is helpful for tests because if the io.Reader contains a json value that does not
// successfully unmarshal into the provided value `v` then the unexpected value contained in the
// io.Reader can be printed for debugging
func UnmarshalReaderJSON(r io.Reader, v interface{}) error {
	b, err := ioutil.ReadAll(r)
	if err != nil {
		return fmt.Errorf("error reading response body: %v", err)
	}
	err = json.Unmarshal(b, v)
	if err != nil {
		return fmt.Errorf("error unmarshaling response body: %v, response body: %s", err, string(b))
	}
	return nil
}

// PrintStatus pretty prints a status code with its' status text
func PrintStatus(code int) string {
	if t := http.StatusText(code); t != "" {
		return fmt.Sprintf("%d %s", code, t)
	}
	return fmt.Sprint(code)
}
