package users

import (
	"errors"
	"fmt"
	"net/http"
	"path"
	"time"

	"code.justin.tv/extensions/eastwatch/internal/api/gql"

	"code.justin.tv/extensions/eastwatch/internal/config"
	"code.justin.tv/extensions/eastwatch/internal/models/token"
)

//go:generate counterfeiter . User

// User is a Twitch user
type User interface {

	// ID is the numeric identifier of a user
	ID() string

	// Name is the user login name
	Name() string

	// OAuthToken retrieves the oauth token that authenticates the user for a given Twitch site
	OAuthToken(config.Configuration, config.Site) (token.OAuth, error)

	// CleanUp cleans up the account and returns it to a clean state
	CleanUp(cfg config.Configuration, h *http.Client, g gql.Client) error
}

var (
	// AnonymousUser is the dummy user for anonymous access
	AnonymousUser = anonymous{}

	// ErrNoOAuthToken indicates that the user's oauth token could not be resolved
	ErrNoOAuthToken = errors.New("user has no oauth token")

	// Dummy Users that can be used in tests
	User1 User

	// Dummy Users that can be used in tests
	User2 User

	// Dummy Users that can be used in tests
	User3 User
)

type anonymous struct{}

func (u anonymous) ID() string   { return "" }
func (u anonymous) Name() string { return "" }
func (u anonymous) OAuthToken(config.Configuration, config.Site) (string, error) {
	return "", ErrNoOAuthToken
}
func (u anonymous) CleanUp() error { return nil }

type sandstormUser struct {
	LoginName string
	AccountID string

	clientIDs   map[config.Site]string
	oauthTokens map[config.Site]token.OAuth

	gqlClient  gql.Client
	httpClient *http.Client

	cfg config.Configuration
}

// NewSandstormUser is a user where the client-id and oauth token are retrieved from sandstorm
func NewSandstormUser(loginName, accountID string, http http.Client, cfg config.Configuration) User {
	g, err := gql.NewClient(&http, cfg.GQLURL)
	if err != nil {
		panic("Unable to initialize GQL Client for users! " + err.Error())
	}
	return &sandstormUser{
		LoginName:   loginName,
		AccountID:   accountID,
		clientIDs:   make(map[config.Site]string),
		oauthTokens: make(map[config.Site]token.OAuth),
		httpClient:  &http,
		cfg:         cfg,
		gqlClient:   g,
	}
}

func (u *sandstormUser) ID() string   { return u.AccountID }
func (u *sandstormUser) Name() string { return u.LoginName }
func (u *sandstormUser) OAuthToken(c config.Configuration, s config.Site) (token.OAuth, error) {
	if t, found := u.oauthTokens[s]; found {
		return t, nil
	}

	if u.LoginName == "" {
		return "", ErrNoOAuthToken
	}

	k := path.Join(c.BaseSandstormPath, fmt.Sprintf("eastwatch_%s_%s_oauth", u.LoginName, s))
	t, err := config.GetSandstormSecret(c.SandstormARN, k)
	if err != nil {
		return "", err
	}

	oa, err := token.NewOAuth(t)
	if err != nil {
		return token.OAuthEmpty, err
	}
	u.oauthTokens[s] = oa
	return oa, nil
}

func init() {
	users := NewUserSet()

	h := http.Client{
		Timeout: 5 * time.Second,
	}

	cfg := config.Staging

	User1 = NewSandstormUser(users[0].loginName, users[0].accountID, h, cfg)
	User2 = NewSandstormUser(users[1].loginName, users[1].accountID, h, cfg)
	User3 = NewSandstormUser(users[2].loginName, users[2].accountID, h, cfg)
}
