package test

import (
	"fmt"
	"net/http"
	"runtime/debug"
	"strings"
	"time"

	"code.justin.tv/extensions/eastwatch/internal/fultonlibs/FultonGoLangSimpleBootstrap/simplebootstrap"
	"code.justin.tv/extensions/eastwatch/internal/metrics"

	"code.justin.tv/extensions/eastwatch/internal/testutil"
	"github.com/stretchr/testify/suite"
)

type EastwatchSuite struct {
	suite.Suite

	Recorder   *testutil.RoundTripRecorder
	HTTPClient *http.Client

	SetupError error

	TearDownSteps  []func() error
	TearDownErrors []error

	Bootstrap *simplebootstrap.FultonBootstrap
}

func printStackTraceOnPanic() {
	t := recover()
	if t != nil {
		debug.PrintStack()
		panic(t)
	}
}

func (s *EastwatchSuite) SetupSuite() {
	defer printStackTraceOnPanic()
	bs, err := metrics.InitBootStrap("Eastwatch")
	if err != nil {
		panic(fmt.Sprintf("unable to initialize stats! %v", err))
	}
	s.Bootstrap = bs
	rt := testutil.NewMetricsRoundTripper(http.DefaultTransport, s.Bootstrap)
	s.Recorder = testutil.NewRoundTripRecorder(rt, s.PrintRoundTripRecord)
	s.HTTPClient = &http.Client{
		Timeout:   5 * time.Second,
		Transport: s.Recorder,
	}
}

func (s *EastwatchSuite) RunSetups(setups ...func() error) error {
	defer printStackTraceOnPanic()
	for _, setup := range setups {
		if err := setup(); err != nil {
			//Retry
			s.T().Logf("Retry Setup Step")
			//Adding buffer between calls to reduce 409 issues
			time.Sleep(time.Duration(250) * time.Millisecond)
			if err2 := setup(); err2 != nil {
				s.SetupError = err2
				s.T().Logf("setup stopped early due to error: %v and %v", err, err2)
				return err2
			}
		}
	}
	return nil
}

// RunTearDowns runs the teardown steps in reverse order that they were added
func (s *EastwatchSuite) RunTearDowns() error {
	defer printStackTraceOnPanic()
	// manually tear down so that the test suite has a chance to capture any values during
	// tear down to be asserted upon
	for i := len(s.TearDownSteps) - 1; i >= 0; i-- {
		tearDown := s.TearDownSteps[i]
		if err := tearDown(); err != nil {
			//Retry
			s.T().Logf("Retry Tear Down Step")
			//Adding buffer between calls to reduce 409 issues
			time.Sleep(time.Duration(250) * time.Millisecond)
			if err := tearDown(); err != nil {
				s.T().Logf("teardown stopped early due to error: %v", err)
				s.TearDownErrors = append(s.TearDownErrors, err)
			}
		}
	}

	// Push metrics to Cloudwatch
	s.Bootstrap.SampleReporter.Stop()

	// TODO: build a composite error??
	if len(s.TearDownErrors) > 0 {
		return s.TearDownErrors[0]
	}
	return nil
}

func (s *EastwatchSuite) AddTearDownStep(tearDown func() error) {
	s.TearDownSteps = append(s.TearDownSteps, tearDown)
}

// PrintRoundTripRecord will output the request, error and response to the console
func (s *EastwatchSuite) PrintRoundTripRecord(r testutil.RoundTripRecord) {
	const header = "\n--- %s\n"
	var sb strings.Builder
	if r.RequestDump != "" {
		sb.WriteString(fmt.Sprintf(header, "REQUEST"))
		sb.WriteString(r.RequestDump)
	}
	if r.ResponseDump != "" {
		sb.WriteString(fmt.Sprintf(header, "RESPONSE"))
		sb.WriteString(r.ResponseDump)
	}
	if r.Error != nil {
		sb.WriteString(fmt.Sprintf(header, "ERROR"))
		sb.WriteString(fmt.Sprint(r.Error))
	}
	if msg := sb.String(); msg != "" {
		s.T().Log(sb.String())
	}
}
